"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssetClient = void 0;
exports.getAssetLinkUuid = getAssetLinkUuid;
var _objectHash = _interopRequireDefault(require("object-hash"));
var _fields = require("./fields");
var _asset_not_found_error = require("../errors/asset_not_found_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function termQuery(field, value, opts = {
  queryEmptyString: true
}) {
  if (value === null || value === undefined || !opts.queryEmptyString && value === '') {
    return [];
  }
  return [{
    term: {
      [field]: value
    }
  }];
}
function termsQuery(field, values) {
  if (values === null || values === undefined || values.length === 0) {
    return [];
  }
  const filteredValues = values.filter(value => value !== undefined);
  return [{
    terms: {
      [field]: filteredValues
    }
  }];
}
function getAssetLinkUuid(name, asset) {
  return (0, _objectHash.default)({
    [_fields.STREAM_NAME]: name,
    [_fields.ASSET_ID]: asset[_fields.ASSET_ID],
    [_fields.ASSET_TYPE]: asset[_fields.ASSET_TYPE]
  });
}
function toAssetLink(name, asset) {
  return {
    ...asset,
    [_fields.ASSET_UUID]: getAssetLinkUuid(name, asset)
  };
}
function fromStorage(link) {
  const storedQueryLink = link;
  return {
    ...storedQueryLink,
    query: {
      id: storedQueryLink[_fields.ASSET_ID],
      title: storedQueryLink[_fields.QUERY_TITLE],
      kql: {
        query: storedQueryLink[_fields.QUERY_KQL_BODY]
      },
      feature: storedQueryLink[_fields.QUERY_FEATURE_NAME] ? {
        name: storedQueryLink[_fields.QUERY_FEATURE_NAME],
        filter: JSON.parse(storedQueryLink[_fields.QUERY_FEATURE_FILTER])
      } : undefined
    }
  };
}
function toStorage(name, request) {
  const link = toAssetLink(name, request);
  const {
    query,
    ...rest
  } = link;
  return {
    ...rest,
    [_fields.STREAM_NAME]: name,
    [_fields.QUERY_TITLE]: query.title,
    [_fields.QUERY_KQL_BODY]: query.kql.query,
    [_fields.QUERY_FEATURE_NAME]: query.feature ? query.feature.name : '',
    [_fields.QUERY_FEATURE_FILTER]: query.feature ? JSON.stringify(query.feature.filter) : ''
  };
}
class AssetClient {
  constructor(clients) {
    this.clients = clients;
  }
  async linkAsset(name, link) {
    const document = toStorage(name, link);
    await this.clients.storageClient.index({
      id: document[_fields.ASSET_UUID],
      document
    });
    return toAssetLink(name, link);
  }
  async syncAssetList(name, links, assetType) {
    const assetsResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: false,
      query: {
        bool: {
          filter: [...termQuery(_fields.STREAM_NAME, name), ...termQuery(_fields.ASSET_TYPE, assetType)]
        }
      }
    });
    const existingAssetLinks = assetsResponse.hits.hits.map(hit => {
      return fromStorage(hit._source);
    });
    const nextAssetLinks = links.map(link => {
      return toAssetLink(name, link);
    });
    const nextIds = new Set(nextAssetLinks.map(link => link[_fields.ASSET_UUID]));
    const assetLinksDeleted = existingAssetLinks.filter(link => !nextIds.has(link[_fields.ASSET_UUID]));
    const operations = [...assetLinksDeleted.map(asset => ({
      delete: {
        asset
      }
    })), ...nextAssetLinks.map(asset => ({
      index: {
        asset
      }
    }))];
    if (operations.length) {
      await this.bulk(name, operations);
    }
    return {
      deleted: assetLinksDeleted,
      indexed: nextAssetLinks
    };
  }
  async unlinkAsset(name, asset) {
    const id = getAssetLinkUuid(name, asset);
    const {
      result
    } = await this.clients.storageClient.delete({
      id
    });
    if (result === 'not_found') {
      throw new _asset_not_found_error.AssetNotFoundError(`${asset[_fields.ASSET_TYPE]} not found`);
    }
  }
  async clean() {
    await this.clients.storageClient.clean();
  }
  async getAssetLinks(names, assetTypes) {
    const filters = [...termsQuery(_fields.STREAM_NAME, names)];
    if (assetTypes !== null && assetTypes !== void 0 && assetTypes.length) {
      filters.push(...termsQuery(_fields.ASSET_TYPE, assetTypes));
    }
    const assetsResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: false,
      query: {
        bool: {
          filter: filters
        }
      }
    });
    const assetsPerName = names.reduce((acc, name) => {
      acc[name] = [];
      return acc;
    }, {});
    assetsResponse.hits.hits.forEach(hit => {
      const name = hit._source[_fields.STREAM_NAME];
      const asset = fromStorage(hit._source);
      assetsPerName[name].push(asset);
    });
    return assetsPerName;
  }
  async bulkGetByIds(name, assetType, ids) {
    const assetsResponse = await this.clients.storageClient.search({
      size: 10_000,
      track_total_hits: false,
      query: {
        bool: {
          filter: [...termQuery(_fields.STREAM_NAME, name), ...termQuery(_fields.ASSET_TYPE, assetType), ...termsQuery('_id', ids.map(id => getAssetLinkUuid(name, {
            [_fields.ASSET_TYPE]: assetType,
            [_fields.ASSET_ID]: id
          })))]
        }
      }
    });
    return assetsResponse.hits.hits.map(hit => fromStorage(hit._source));
  }
  async bulk(name, operations) {
    return await this.clients.storageClient.bulk({
      operations: operations.map(operation => {
        if ('index' in operation) {
          const document = toStorage(name, Object.values(operation)[0].asset);
          return {
            index: {
              document,
              _id: document[_fields.ASSET_UUID]
            }
          };
        }
        const id = getAssetLinkUuid(name, operation.delete.asset);
        return {
          delete: {
            _id: id
          }
        };
      }),
      throwOnFail: true
    });
  }
  async getAssets(name) {
    const {
      [name]: assetLinks
    } = await this.getAssetLinks([name]);
    if (assetLinks.length === 0) {
      return [];
    }
    return assetLinks.filter(link => link[_fields.ASSET_TYPE] === 'query').map(link => {
      return {
        [_fields.ASSET_ID]: link[_fields.ASSET_ID],
        [_fields.ASSET_UUID]: link[_fields.ASSET_UUID],
        [_fields.ASSET_TYPE]: link[_fields.ASSET_TYPE],
        query: link.query,
        title: link.query.title
      };
    });
  }
}
exports.AssetClient = AssetClient;