"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.showErrorToast = showErrorToast;
exports.useStreamsAppFetch = useStreamsAppFetch;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _serverRouteRepositoryClient = require("@kbn/server-route-repository-client");
var _reactHooks = require("@kbn/react-hooks");
var _use_kibana = require("./use_kibana");
var _use_timefilter = require("./use_timefilter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useStreamsAppFetch(callback, deps, options) {
  const {
    core: {
      notifications
    }
  } = (0, _use_kibana.useKibana)();
  const {
    disableToastOnError = false,
    withRefresh = false,
    withTimeRange = false
  } = options || {};
  const {
    timeState,
    timeState$
  } = (0, _use_timefilter.useTimefilter)();
  const onError = error => {
    if (!disableToastOnError && !(0, _serverRouteRepositoryClient.isRequestAbortedError)(error)) {
      showErrorToast(notifications, error);

      // log to console to get the actual stack trace
      // eslint-disable-next-line no-console
      console.log(error);
    }
  };
  const optionsForHook = {
    ...(0, _lodash.omit)(options, 'disableToastOnError', 'withTimeRange'),
    onError
  };
  const timeStateRef = (0, _react.useRef)();
  timeStateRef.current = timeState;
  const state = (0, _reactHooks.useAbortableAsync)(({
    signal
  }) => {
    const parameters = {
      signal,
      ...(withTimeRange ? {
        timeState: timeStateRef.current
      } : {})
    };
    return callback(parameters);
  },
  // eslint-disable-next-line react-hooks/exhaustive-deps
  deps, optionsForHook);
  const refreshRef = (0, _react.useRef)(state.refresh);
  refreshRef.current = state.refresh;
  (0, _react.useEffect)(() => {
    const subscription = timeState$.subscribe({
      next: ({
        kind
      }) => {
        const shouldRefresh = withTimeRange && kind === 'shift' || withRefresh && kind !== 'initial';
        if (shouldRefresh) {
          refreshRef.current();
        }
      }
    });
    return () => {
      subscription.unsubscribe();
    };
  }, [timeState$, withTimeRange, withRefresh]);
  return state;
}
function showErrorToast(notifications, error) {
  if ('body' in error && typeof error.body === 'object' && !!error.body && 'message' in error.body && typeof error.body.message === 'string') {
    error.message = error.body.message;
  }
  if (error instanceof AggregateError) {
    error.message = error.errors.map(err => err.message).join(', ');
  }
  let requestUrl;
  if ('request' in error && typeof error.request === 'object' && !!error.request && 'url' in error.request && typeof error.request.url === 'string') {
    requestUrl = error.request.url;
  }
  return notifications.toasts.addError(error, {
    title: _i18n.i18n.translate('xpack.streams.failedToFetchError', {
      defaultMessage: 'Failed to fetch data{requestUrlSuffix}',
      values: {
        requestUrlSuffix: requestUrl ? ` (${requestUrl})` : ''
      }
    })
  });
}