"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useStreamFeaturesApi = useStreamFeaturesApi;
var _reactHooks = require("@kbn/react-hooks");
var _rxjs = require("rxjs");
var _use_kibana = require("./use_kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useStreamFeaturesApi(definition) {
  const {
    dependencies: {
      start: {
        streams: {
          streamsRepositoryClient
        }
      }
    }
  } = (0, _use_kibana.useKibana)();
  const {
    signal,
    abort,
    refresh
  } = (0, _reactHooks.useAbortController)();
  return {
    identifyFeatures: async (connectorId, to, from) => {
      const events$ = streamsRepositoryClient.stream('POST /internal/streams/{name}/features/_identify', {
        signal,
        params: {
          path: {
            name: definition.name
          },
          query: {
            connectorId,
            to,
            from
          }
        }
      });
      return (0, _rxjs.firstValueFrom)(events$);
    },
    addFeaturesToStream: async features => {
      return await streamsRepositoryClient.fetch('POST /internal/streams/{name}/features/_bulk', {
        signal,
        params: {
          path: {
            name: definition.name
          },
          body: {
            operations: features.map(feature => ({
              index: {
                feature
              }
            }))
          }
        }
      });
    },
    removeFeaturesFromStream: async features => {
      return await streamsRepositoryClient.fetch('POST /internal/streams/{name}/features/_bulk', {
        signal,
        params: {
          path: {
            name: definition.name
          },
          body: {
            operations: features.map(feature => ({
              delete: {
                feature: {
                  type: feature.type,
                  name: feature.name
                }
              }
            }))
          }
        }
      });
    },
    upsertQuery: async feature => {
      await streamsRepositoryClient.fetch('PUT /internal/streams/{name}/features/{featureType}/{featureName}', {
        signal,
        params: {
          path: {
            name: definition.name,
            featureType: feature.type,
            featureName: feature.name
          },
          body: feature
        }
      });
    },
    abort: () => {
      abort();
      refresh();
    }
  };
}