"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useStreamDataViewFieldTypes = useStreamDataViewFieldTypes;
var _react = require("react");
var _reactHooks = require("@kbn/react-hooks");
var _use_kibana = require("./use_kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches DataView field types for a stream.
 *
 * @param streamName - The name of the stream to fetch field types for
 * @returns Object containing field types map and DataView
 */
function useStreamDataViewFieldTypes(streamName) {
  const {
    dependencies
  } = (0, _use_kibana.useKibana)();
  const {
    data
  } = dependencies.start;
  const {
    value: dataView
  } = (0, _reactHooks.useAbortableAsync)(async ({
    signal
  }) => {
    try {
      return await data.dataViews.create({
        title: streamName,
        timeFieldName: '@timestamp'
      }, undefined, false);
    } catch (err) {
      // Silently handle errors for new streams that don't have indices yet
      return undefined;
    }
  }, [data.dataViews, streamName]);
  const fieldTypes = (0, _react.useMemo)(() => {
    if (!dataView) {
      return undefined;
    }
    return dataView.fields.map(field => {
      var _field$esTypes;
      return {
        name: field.name,
        type: field.type,
        esType: (_field$esTypes = field.esTypes) === null || _field$esTypes === void 0 ? void 0 : _field$esTypes[0]
      };
    });
  }, [dataView]);
  const fieldTypeMap = (0, _react.useMemo)(() => {
    const typeMap = new Map();
    if (fieldTypes) {
      fieldTypes.forEach(field => {
        const fieldType = field.esType || field.type;
        if (fieldType) {
          typeMap.set(field.name, fieldType);
        }
      });
    }
    return typeMap;
  }, [fieldTypes]);
  return {
    fieldTypes,
    fieldTypeMap,
    dataView
  };
}