"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGenAIConnectors = useGenAIConnectors;
var _react = require("react");
var _useLocalStorage = _interopRequireDefault(require("react-use/lib/useLocalStorage"));
var _managementSettingsIds = require("@kbn/management-settings-ids");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const STREAMS_CONNECTOR_STORAGE_KEY = 'xpack.streamsApp.lastUsedConnector';
const OLD_STORAGE_KEY = 'xpack.observabilityAiAssistant.lastUsedConnector';
// TODO: Import from gen-ai-settings-plugin (package) once available
const NO_DEFAULT_CONNECTOR = 'NO_DEFAULT_CONNECTOR';
function useGenAIConnectors({
  streamsRepositoryClient,
  uiSettings
}) {
  var _connectors$;
  const [connectors, setConnectors] = (0, _react.useState)();
  const [loading, setLoading] = (0, _react.useState)(true);
  const [error, setError] = (0, _react.useState)();

  // Read settings
  const defaultConnector = uiSettings.get(_managementSettingsIds.GEN_AI_SETTINGS_DEFAULT_AI_CONNECTOR);
  const genAISettingsDefaultOnly = uiSettings.get(_managementSettingsIds.GEN_AI_SETTINGS_DEFAULT_AI_CONNECTOR_DEFAULT_ONLY, false);
  const isConnectorSelectionRestricted = genAISettingsDefaultOnly && defaultConnector !== NO_DEFAULT_CONNECTOR;

  // Read old localStorage key (for backward compatibility, don't modify it)
  const [oldConnector] = (0, _useLocalStorage.default)(OLD_STORAGE_KEY);

  // Use old connector as initial value for new key (only if new key doesn't exist yet)
  const [lastUsedConnector, setLastUsedConnector] = (0, _useLocalStorage.default)(STREAMS_CONNECTOR_STORAGE_KEY, oldConnector);
  const fetchConnectors = (0, _react.useCallback)(async () => {
    setLoading(true);
    setError(undefined);
    try {
      const controller = new AbortController();
      const response = await streamsRepositoryClient.fetch('GET /internal/streams/connectors', {
        signal: controller.signal
      });
      let results = response.connectors;

      // If connector selection is restricted, only return the default connector
      if (isConnectorSelectionRestricted) {
        const defaultC = results.find(con => con.id === defaultConnector);
        results = defaultC ? [defaultC] : [];
      }
      setConnectors(results);

      // Clear lastUsedConnector if it's no longer in the list
      setLastUsedConnector(connectorId => {
        if (connectorId && results.findIndex(result => result.id === connectorId) === -1) {
          return undefined;
        }
        return connectorId;
      });
    } catch (err) {
      setError(err);
      setConnectors(undefined);
    } finally {
      setLoading(false);
    }
  }, [streamsRepositoryClient, isConnectorSelectionRestricted, defaultConnector, setLastUsedConnector]);
  (0, _react.useEffect)(() => {
    fetchConnectors();
  }, [fetchConnectors]);

  // Determine selected connector (follows observability pattern)
  const selectedConnector = (0, _react.useMemo)(() => {
    // If restricted, always use default
    if (isConnectorSelectionRestricted) {
      return defaultConnector;
    }

    // Priority 1: User's explicit choice (localStorage)
    if (lastUsedConnector) {
      return lastUsedConnector;
    }

    // Priority 2: Global AI default setting
    if (defaultConnector !== NO_DEFAULT_CONNECTOR) {
      return defaultConnector;
    }
    return undefined;
  }, [isConnectorSelectionRestricted, defaultConnector, lastUsedConnector]);
  const selectConnector = (0, _react.useCallback)(id => {
    setLastUsedConnector(id);
  }, [setLastUsedConnector]);
  const reloadConnectors = (0, _react.useCallback)(async () => {
    await fetchConnectors();
  }, [fetchConnectors]);
  return {
    connectors,
    selectedConnector: selectedConnector || (connectors === null || connectors === void 0 ? void 0 : (_connectors$ = connectors[0]) === null || _connectors$ === void 0 ? void 0 : _connectors$.id),
    loading,
    error,
    selectConnector,
    reloadConnectors,
    isConnectorSelectionRestricted,
    defaultConnector: defaultConnector === NO_DEFAULT_CONNECTOR ? undefined : defaultConnector
  };
}