"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useStreamDescriptionApi = void 0;
var _react = require("react");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _reactHooks = require("@kbn/react-hooks");
var _use_ai_features = require("../../stream_detail_significant_events_view/add_significant_event_flyout/generated_flow_form/use_ai_features");
var _errors = require("../../../util/errors");
var _use_update_streams = require("../../../hooks/use_update_streams");
var _use_kibana = require("../../../hooks/use_kibana");
var _use_timefilter = require("../../../hooks/use_timefilter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useStreamDescriptionApi = ({
  definition,
  refreshDefinition
}) => {
  const {
    signal
  } = (0, _reactHooks.useAbortController)();
  const updateStream = (0, _use_update_streams.useUpdateStreams)(definition.stream.name);
  const aiFeatures = (0, _use_ai_features.useAIFeatures)();
  const {
    core: {
      notifications
    },
    dependencies: {
      start: {
        streams
      }
    }
  } = (0, _use_kibana.useKibana)();
  const {
    timeState
  } = (0, _use_timefilter.useTimefilter)();
  const [description, setDescription] = (0, _react.useState)(definition.stream.description || '');
  const [isGenerating, setIsGenerating] = (0, _react.useState)(false);
  const [isUpdating, setIsUpdating] = (0, _react.useState)(false);
  const [isEditing, setIsEditing] = (0, _react.useState)(false);
  // Save the updated description; show success and error toasts
  const save = (0, _react.useCallback)(async nextDescription => {
    setIsUpdating(true);
    updateStream(_streamsSchema.Streams.all.UpsertRequest.parse({
      dashboards: definition.dashboards,
      queries: definition.queries,
      rules: definition.rules,
      stream: {
        ...(0, _lodash.omit)(definition.stream, 'name'),
        description: nextDescription
      }
    })).then(() => {
      notifications.toasts.addSuccess({
        title: _i18n.i18n.translate('xpack.streams.streamDetailView.streamDescription.saveSuccessTitle', {
          defaultMessage: 'Description saved'
        })
      });
    }).catch(error => {
      notifications.toasts.addError(error, {
        title: _i18n.i18n.translate('xpack.streams.streamDetailView.streamDescription.saveErrorTitle', {
          defaultMessage: 'Failed to save description'
        }),
        toastMessage: (0, _errors.getFormattedError)(error).message
      });
    }).finally(() => {
      setIsUpdating(false);
      refreshDefinition();
    });
  }, [updateStream, definition.dashboards, definition.queries, definition.rules, definition.stream, notifications.toasts, refreshDefinition]);
  const generate = (0, _react.useCallback)(() => {
    if (!(aiFeatures !== null && aiFeatures !== void 0 && aiFeatures.genAiConnectors.selectedConnector)) {
      return;
    }
    setIsGenerating(true);
    streams.streamsRepositoryClient.stream('POST /internal/streams/{name}/_describe_stream', {
      signal,
      params: {
        path: {
          name: definition.stream.name
        },
        query: {
          connectorId: aiFeatures.genAiConnectors.selectedConnector,
          from: timeState.asAbsoluteTimeRange.from,
          to: timeState.asAbsoluteTimeRange.to
        }
      }
    }).subscribe({
      next({
        description: generatedDescription
      }) {
        setDescription(generatedDescription);
      },
      complete() {
        setIsGenerating(false);
      },
      error(error) {
        setIsGenerating(false);
        notifications.toasts.addError(error, {
          title: _i18n.i18n.translate('xpack.streams.streamDetailView.streamDescription.generateErrorTitle', {
            defaultMessage: 'Failed to generate description'
          }),
          toastMessage: (0, _errors.getFormattedError)(error).message
        });
      }
    });
  }, [definition.stream.name, streams.streamsRepositoryClient, timeState.asAbsoluteTimeRange.from, timeState.asAbsoluteTimeRange.to, aiFeatures === null || aiFeatures === void 0 ? void 0 : aiFeatures.genAiConnectors.selectedConnector, signal, notifications.toasts]);
  const onCancelEdit = (0, _react.useCallback)(() => {
    setDescription(definition.stream.description);
    setIsEditing(false);
  }, [setIsEditing, setDescription, definition.stream.description]);
  const onGenerateDescription = (0, _react.useCallback)(() => {
    generate();
    setIsEditing(true);
  }, [generate, setIsEditing]);
  const onStartEditing = (0, _react.useCallback)(() => {
    setIsEditing(true);
  }, [setIsEditing]);
  const onSaveDescription = (0, _react.useCallback)(() => {
    if (description !== definition.stream.description) {
      save(description);
    }
    setIsEditing(false);
  }, [description, definition.stream.description, save, setIsEditing]);
  const areButtonsDisabled = isUpdating || isGenerating;
  return {
    description,
    setDescription,
    isGenerating,
    isUpdating,
    isEditing,
    areButtonsDisabled,
    onCancelEdit,
    onGenerateDescription,
    onStartEditing,
    onSaveDescription
  };
};
exports.useStreamDescriptionApi = useStreamDescriptionApi;