"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractMessagesFromField = extractMessagesFromField;
exports.prepareSamplesForPatternExtraction = prepareSamplesForPatternExtraction;
exports.usePatternSuggestionDependencies = usePatternSuggestionDependencies;
var _objectUtils = require("@kbn/object-utils");
var _lodash = require("lodash");
var _reactHooks = require("@kbn/react-hooks");
var _use_kibana = require("../../../../../../../hooks/use_kibana");
var _selectors = require("../../../../state_management/simulation_state_machine/selectors");
var _stream_enrichment_state_machine = require("../../../../state_management/stream_enrichment_state_machine");
var _simulation_runner_actor = require("../../../../state_management/simulation_state_machine/simulation_runner_actor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Prepares samples for pattern extraction by:
 * 1. Flattening original samples
 * 2. Optionally running partial simulation if there are previous processing steps
 * 3. Applying preview document filters
 */
async function prepareSamplesForPatternExtraction(originalSamples, stepsWithoutCurrent, previewDocsFilter, streamsRepositoryClient, streamName) {
  let samples = originalSamples.map(doc => doc.document).map(_objectUtils.flattenObjectNestedLast);

  /**
   * If there are processors, we run a partial simulation to get the samples.
   * If there are no processors, we use the original samples previously assigned.
   */
  if (stepsWithoutCurrent.length > 0) {
    const simulation = await (0, _simulation_runner_actor.simulateProcessing)({
      streamsRepositoryClient,
      input: {
        streamName,
        steps: stepsWithoutCurrent,
        documents: samples
      }
    });
    samples = (0, _selectors.selectPreviewRecords)({
      samples: originalSamples,
      previewDocsFilter,
      simulation
    });
  }
  return samples;
}

/**
 * Extracts string messages from a specific field in the samples.
 * Filters out non-string values.
 */
function extractMessagesFromField(samples, fieldName) {
  return samples.reduce((acc, sample) => {
    const value = (0, _lodash.get)(sample, fieldName);
    if (typeof value === 'string') {
      acc.push(value);
    }
    return acc;
  }, []);
}

/**
 * Custom hook that provides common dependencies needed for pattern suggestions.
 * Returns Kibana services, abort controller, and simulator state.
 */
function usePatternSuggestionDependencies() {
  const {
    core: {
      notifications
    },
    services: {
      telemetryClient
    },
    dependencies: {
      start: {
        streams: {
          streamsRepositoryClient
        }
      }
    }
  } = (0, _use_kibana.useKibana)();
  const abortController = (0, _reactHooks.useAbortController)();
  const stepsWithoutCurrent = (0, _stream_enrichment_state_machine.useSimulatorSelector)(snapshot => snapshot.context.steps.slice(0, -1));
  const previewDocsFilter = (0, _stream_enrichment_state_machine.useSimulatorSelector)(snapshot => snapshot.context.previewDocsFilter);
  const originalSamples = (0, _stream_enrichment_state_machine.useSimulatorSelector)(snapshot => (0, _selectors.selectOriginalPreviewRecords)(snapshot.context));
  return {
    notifications,
    telemetryClient,
    streamsRepositoryClient,
    abortController,
    stepsWithoutCurrent,
    previewDocsFilter,
    originalSamples
  };
}