"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SUGGESTED_GROK_PROCESSOR_ID = void 0;
exports.useGrokPatternSuggestion = useGrokPatternSuggestion;
var _useAsyncFn = _interopRequireDefault(require("react-use/lib/useAsyncFn"));
var _grokHeuristics = require("@kbn/grok-heuristics");
var _rxjs = require("rxjs");
var _use_streams_app_fetch = require("../../../../../../../hooks/use_streams_app_fetch");
var _pattern_suggestion_helpers = require("../utils/pattern_suggestion_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SUGGESTED_GROK_PROCESSOR_ID = exports.SUGGESTED_GROK_PROCESSOR_ID = 'grok-processor';
function useGrokPatternSuggestion() {
  const {
    notifications,
    telemetryClient,
    streamsRepositoryClient,
    abortController,
    stepsWithoutCurrent,
    previewDocsFilter,
    originalSamples
  } = (0, _pattern_suggestion_helpers.usePatternSuggestionDependencies)();
  return (0, _useAsyncFn.default)(async params => {
    if (params === null) {
      return Promise.resolve(undefined); // Reset to initial value
    }

    // Prepare samples by running partial simulation if needed
    const samples = await (0, _pattern_suggestion_helpers.prepareSamplesForPatternExtraction)(originalSamples, stepsWithoutCurrent, previewDocsFilter, streamsRepositoryClient, params.streamName);

    // Extract string messages from the target field
    const messages = (0, _pattern_suggestion_helpers.extractMessagesFromField)(samples, params.fieldName);
    const groupedMessages = (0, _grokHeuristics.groupMessagesByPattern)(messages);
    const finishTrackingAndReport = telemetryClient.startTrackingAIGrokSuggestionLatency({
      name: params.streamName,
      field: params.fieldName,
      connector_id: params.connectorId
    });
    const result = await Promise.allSettled(groupedMessages.map(group => {
      const grokPatternNodes = (0, _grokHeuristics.extractGrokPatternDangerouslySlow)(group.messages);

      // The only reason we're streaming the response here is to avoid timeout issues prevalent with long-running requests to LLMs.
      // There is only ever going to be a single event emitted so we can safely use `lastValueFrom`.
      return (0, _rxjs.lastValueFrom)(streamsRepositoryClient.stream('POST /internal/streams/{name}/processing/_suggestions/grok', {
        signal: abortController.signal,
        params: {
          path: {
            name: params.streamName
          },
          body: {
            connector_id: params.connectorId,
            sample_messages: group.messages.slice(0, 10),
            review_fields: (0, _grokHeuristics.getReviewFields)(grokPatternNodes, 10)
          }
        }
      })).then(reviewResult => {
        const grokProcessor = (0, _grokHeuristics.getGrokProcessor)(grokPatternNodes, reviewResult.grokProcessor);
        return {
          ...grokProcessor,
          patterns: (0, _grokHeuristics.unwrapPatternDefinitions)(grokProcessor),
          // NOTE: Inline patterns until we support custom pattern definitions in Streamlang
          pattern_definitions: {}
        };
      });
    }));
    const aggregateError = new AggregateError(result.reduce((acc, settledState) => {
      if (settledState.status === 'rejected') {
        acc.push(settledState.reason);
      }
      return acc;
    }, []));
    const grokProcessors = result.reduce((acc, settledState) => {
      if (settledState.status === 'fulfilled') {
        acc.push(settledState.value);
      }
      return acc;
    }, []);

    // If all promises failed, throw an aggregate error, otherwise ignore errors and continue with fulfilled results
    if (grokProcessors.length === 0) {
      finishTrackingAndReport(0, [0]);
      (0, _use_streams_app_fetch.showErrorToast)(notifications, aggregateError);
      throw aggregateError;
    }

    // Combine all grok processors into a single one with fallback patterns
    const combinedGrokProcessor = (0, _grokHeuristics.mergeGrokProcessors)(grokProcessors);

    // Run simulation to get fields and metrics
    const simulationResult = await streamsRepositoryClient.fetch('POST /internal/streams/{name}/processing/_simulate', {
      signal: abortController.signal,
      params: {
        path: {
          name: params.streamName
        },
        body: {
          documents: samples,
          processing: {
            steps: [{
              action: 'grok',
              customIdentifier: SUGGESTED_GROK_PROCESSOR_ID,
              from: params.fieldName,
              patterns: combinedGrokProcessor.patterns
            }]
          }
        }
      }
    });
    finishTrackingAndReport(1, [simulationResult.processors_metrics[SUGGESTED_GROK_PROCESSOR_ID].parsed_rate]);
    return {
      grokProcessor: combinedGrokProcessor,
      simulationResult
    };
  }, [abortController, stepsWithoutCurrent, previewDocsFilter, originalSamples, notifications, streamsRepositoryClient, telemetryClient]);
}