"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.collectDescendantIds = collectDescendantIds;
exports.defaultLatestSamplesDataSource = exports.defaultKqlSamplesDataSource = exports.defaultEnrichmentUrlState = exports.createDefaultCustomSamplesDataSource = void 0;
exports.findInsertIndex = findInsertIndex;
exports.getActiveDataSourceRef = getActiveDataSourceRef;
exports.getActiveDataSourceSamples = getActiveDataSourceSamples;
exports.getActiveSimulationMode = getActiveSimulationMode;
exports.getConfiguredSteps = getConfiguredSteps;
exports.getDataSourcesUrlState = getDataSourcesUrlState;
exports.getRootLevelStepsMap = getRootLevelStepsMap;
exports.getStepsForSimulation = getStepsForSimulation;
exports.getUpsertFields = getUpsertFields;
exports.insertAtIndex = insertAtIndex;
exports.reorderSteps = reorderSteps;
exports.selectDataSource = selectDataSource;
exports.spawnStep = exports.spawnDataSource = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _streamlang = require("@kbn/streamlang/types/streamlang");
var _uuid = require("uuid");
var _common = require("../../../../../../common/url_schema/common");
var _simulation_state_machine = require("../simulation_state_machine");
var _utils = require("../../utils");
var _steps_state_machine = require("../steps_state_machine");
var _translations = require("../../data_sources_flyout/translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultLatestSamplesDataSource = exports.defaultLatestSamplesDataSource = {
  type: 'latest-samples',
  name: _translations.DATA_SOURCES_I18N.latestSamples.defaultName,
  enabled: true
};
const defaultKqlSamplesDataSource = exports.defaultKqlSamplesDataSource = {
  type: 'kql-samples',
  name: _translations.DATA_SOURCES_I18N.kqlDataSource.defaultName,
  enabled: true,
  timeRange: {
    from: 'now-15m',
    to: 'now'
  },
  filters: [],
  query: {
    language: 'kuery',
    query: ''
  }
};
const createDefaultCustomSamplesDataSource = streamName => ({
  type: 'custom-samples',
  name: _translations.DATA_SOURCES_I18N.customSamples.defaultName,
  enabled: true,
  documents: [],
  storageKey: `${_common.CUSTOM_SAMPLES_DATA_SOURCE_STORAGE_KEY_PREFIX}${streamName}__${(0, _uuid.v4)()}`
});
exports.createDefaultCustomSamplesDataSource = createDefaultCustomSamplesDataSource;
const defaultEnrichmentUrlState = exports.defaultEnrichmentUrlState = {
  v: 1,
  dataSources: [defaultLatestSamplesDataSource]
};
function getDataSourcesUrlState(context) {
  const dataSources = context.dataSourcesRefs.map(dataSourceRef => dataSourceRef.getSnapshot().context.dataSource);
  return dataSources.map(dataSource =>
  // Don't persist custom samples documents
  dataSource.type === 'custom-samples' ? {
    ...dataSource,
    documents: []
  } : dataSource).map(_utils.dataSourceConverter.toUrlSchema);
}
function getActiveDataSourceSamples(context) {
  const dataSourceSnapshot = context.dataSourcesRefs.map(dataSourceRef => dataSourceRef.getSnapshot()).find(snapshot => snapshot.matches('enabled'));
  if (!dataSourceSnapshot) return [];
  return dataSourceSnapshot.context.data.map(doc => ({
    dataSourceId: dataSourceSnapshot.context.dataSource.id,
    document: doc
  }));
}

/**
 * Gets processors for simulation based on current editing state.
 * - If no processor is being edited: returns all new processors
 * - If a processor is being edited: returns new processors up to and including the one being edited
 */
function getStepsForSimulation({
  stepRefs,
  isPartialSimulation
}) {
  let newStepSnapshots = stepRefs.map(procRef => procRef.getSnapshot()).filter(snapshot => (0, _streamlang.isWhereBlock)(snapshot.context.step) || (isPartialSimulation ? snapshot.context.isNew : true));

  // Find if any processor is currently being edited
  const editingProcessorIndex = newStepSnapshots.findIndex(snapshot => (0, _streamlang.isActionBlock)(snapshot.context.step) && (0, _steps_state_machine.isStepUnderEdit)(snapshot));

  // If a processor is being edited, set new processors up to and including the one being edited
  if (editingProcessorIndex !== -1) {
    newStepSnapshots = newStepSnapshots.slice(0, editingProcessorIndex + 1);
  }

  // Return processors
  return newStepSnapshots.map(snapshot => snapshot.context.step);
}
function getConfiguredSteps(context) {
  return context.stepRefs.map(proc => proc.getSnapshot()).filter(proc => proc.matches('configured')).map(proc => proc.context.step);
}
function getUpsertFields(context) {
  if (!context.simulatorRef) {
    return undefined;
  }
  const originalFieldDefinition = {
    ...(_streamsSchema.Streams.WiredStream.GetResponse.is(context.definition) ? context.definition.stream.ingest.wired.fields : context.definition.stream.ingest.classic.field_overrides)
  };
  const {
    detectedSchemaFields
  } = context.simulatorRef.getSnapshot().context;

  // Remove unmapped fields from original definition
  const unmappedSchemaFields = (0, _simulation_state_machine.getUnmappedSchemaFields)(detectedSchemaFields);
  unmappedSchemaFields.forEach(field => {
    delete originalFieldDefinition[field.name];
  });
  const mappedSchemaFields = (0, _simulation_state_machine.getMappedSchemaFields)(detectedSchemaFields);
  const simulationMappedFieldDefinition = (0, _simulation_state_machine.convertToFieldDefinition)(mappedSchemaFields);
  return {
    ...originalFieldDefinition,
    ...simulationMappedFieldDefinition
  };
}
const spawnStep = (step, assignArgs, options) => {
  var _options$isNew;
  const {
    spawn,
    self
  } = assignArgs;
  return spawn('stepMachine', {
    id: step.customIdentifier,
    input: {
      parentRef: self,
      step,
      isNew: (_options$isNew = options === null || options === void 0 ? void 0 : options.isNew) !== null && _options$isNew !== void 0 ? _options$isNew : false
    }
  });
};
exports.spawnStep = spawnStep;
const spawnDataSource = (dataSource, assignArgs) => {
  const {
    spawn,
    context,
    self
  } = assignArgs;
  const dataSourceWithUIAttributes = _utils.dataSourceConverter.toUIDefinition(dataSource);
  return spawn('dataSourceMachine', {
    id: dataSourceWithUIAttributes.id,
    input: {
      parentRef: self,
      streamName: context.definition.stream.name,
      dataSource: dataSourceWithUIAttributes
    }
  });
};

/* Find insert index based on step hierarchy */
exports.spawnDataSource = spawnDataSource;
function findInsertIndex(stepRefs, parentId) {
  // Find the index of the parent step
  const parentIndex = parentId ? stepRefs.findIndex(step => step.id === parentId) : -1;

  // Find the last index of any step with the same parentId
  let lastSiblingIndex = -1;
  if (parentId !== null) {
    for (let i = 0; i < stepRefs.length; i++) {
      if (stepRefs[i].getSnapshot().context.step.parentId === parentId) {
        lastSiblingIndex = i;
      }
    }
  }
  if (lastSiblingIndex !== -1) {
    // Insert after the last sibling with the same parentId
    return lastSiblingIndex + 1;
  } else if (parentIndex !== -1) {
    // Insert right after the parent if no siblings
    return parentIndex + 1;
  } else {
    // No parent, insert at the end
    return stepRefs.length;
  }
}
function insertAtIndex(array, item, index) {
  return [...array.slice(0, index), item, ...array.slice(index)];
}

/**
 * Moves a contiguous block of steps (step + all descendants) up or down in the steps array.
 * This means reordering around other contiguous blocks of steps.
 * Maintains a strict index-based ordering of steps that reflects the hierarchy.
 * @param stepRefs The flat array of StepActorRef
 * @param stepId The customIdentifier of the root step to move
 * @param direction 'up' or 'down'
 * @returns A new reordered array of StepActorRef
 */
function reorderSteps(stepRefs, stepId, direction) {
  // 1. Collect all descendant ids for the block to move
  const children = collectDescendantIds(stepId, stepRefs);
  const allBlockIds = new Set([stepId, ...children]);

  // 2. Find the start and end index of the block in the original array
  const startIndex = stepRefs.findIndex(step => step.id === stepId);
  const lastChildId = Array.from(children).pop();
  const endIndex = lastChildId ? stepRefs.findIndex(step => step.id === lastChildId) + 1 : startIndex + 1;
  const block = stepRefs.slice(startIndex, endIndex);

  // 3. Remove the block from the array
  const withoutBlock = [...stepRefs.slice(0, startIndex), ...stepRefs.slice(endIndex)];

  // 4. Get the parentId of the block root
  const blockRootParentId = stepRefs[startIndex].getSnapshot().context.step.parentId;
  if (direction === 'up') {
    // Find the previous block with the same parentId
    let insertIndex = 0;
    for (let i = startIndex - 1; i >= 0; i--) {
      const candidate = stepRefs[i];
      const candidateStep = candidate.getSnapshot().context.step;
      if (!allBlockIds.has(candidate.id) && candidateStep.parentId === blockRootParentId) {
        // Find the start of this previous block in withoutBlock
        const candidateBlockStart = withoutBlock.findIndex(step => step.id === candidate.id);
        insertIndex = candidateBlockStart;
        break;
      }
    }
    // If not found, insert at the top among siblings
    return [...withoutBlock.slice(0, insertIndex), ...block, ...withoutBlock.slice(insertIndex)];
  } else {
    // direction === 'down'
    // Find the next block with the same parentId
    let insertIndex = withoutBlock.length;
    for (let i = endIndex; i < stepRefs.length; i++) {
      const candidate = stepRefs[i];
      const candidateStep = candidate.getSnapshot().context.step;
      if (!allBlockIds.has(candidate.id) && candidateStep.parentId === blockRootParentId) {
        // Find the end of this next block in withoutBlock
        let candidateBlockEnd = withoutBlock.findIndex(step => step.id === candidate.id);
        // Find the last descendant of this block
        const candidateDescendants = collectDescendantIds(candidate.id, stepRefs);
        if (candidateDescendants.size > 0) {
          const lastDescendantId = Array.from(candidateDescendants).pop();
          const lastDescendantIdx = withoutBlock.findIndex(step => step.id === lastDescendantId);
          if (lastDescendantIdx !== -1) {
            candidateBlockEnd = lastDescendantIdx;
          }
        }
        insertIndex = candidateBlockEnd + 1;
        break;
      }
    }
    // If not found, insert at the end among siblings
    return [...withoutBlock.slice(0, insertIndex), ...block, ...withoutBlock.slice(insertIndex)];
  }
}
function collectDescendantIds(id, stepRefs) {
  const ids = new Set();
  function collect(currentId) {
    stepRefs.filter(step => step.getSnapshot().context.step.parentId === currentId).forEach(child => {
      ids.add(child.id);
      collect(child.id);
    });
  }
  collect(id);
  return ids;
}
// Maps every step to their corresponding root level step
function getRootLevelStepsMap(stepRefs) {
  // Build a lookup for quick access to parentId by customIdentifier
  const idToParentId = new Map();
  for (const ref of stepRefs) {
    const {
      customIdentifier,
      parentId
    } = ref.getSnapshot().context.step;
    idToParentId.set(customIdentifier, parentId !== null && parentId !== void 0 ? parentId : null);
  }

  // For each step, walk up the parent chain to find the root ancestor
  const result = new Map();
  for (const ref of stepRefs) {
    const currentId = ref.getSnapshot().context.step.customIdentifier;
    let parentId = idToParentId.get(currentId);

    // Walk up until we find a step with no parentId (root)
    let rootId = currentId;
    while (parentId) {
      var _idToParentId$get;
      rootId = parentId;
      parentId = (_idToParentId$get = idToParentId.get(rootId)) !== null && _idToParentId$get !== void 0 ? _idToParentId$get : null;
    }
    result.set(currentId, rootId);
  }
  return result;
}
function getActiveDataSourceRef(dataSourcesRefs) {
  return dataSourcesRefs.find(dataSourceRef => dataSourceRef.getSnapshot().matches('enabled'));
}
function getActiveSimulationMode(context) {
  const activeDataSourceRef = getActiveDataSourceRef(context.dataSourcesRefs);
  if (!activeDataSourceRef) return 'partial';
  return activeDataSourceRef.getSnapshot().context.simulationMode;
}
function selectDataSource(dataSourcesRefs, id) {
  dataSourcesRefs.forEach(dataSourceRef => {
    if (dataSourceRef.id === id) {
      dataSourceRef.send({
        type: 'dataSource.enable'
      });
    } else {
      dataSourceRef.send({
        type: 'dataSource.disable'
      });
    }
  });
}