"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.simulationMachine = exports.createSimulationMachineImplementations = void 0;
var _xstate = require("xstate5");
var _xstateUtils = require("@kbn/xstate-utils");
var _lodash = require("lodash");
var _objectUtils = require("@kbn/object-utils");
var _utils = require("../../utils");
var _simulation_runner_actor = require("./simulation_runner_actor");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const hasSamples = samples => !(0, _lodash.isEmpty)(samples);
const hasAnyValidSteps = steps => {
  const validSteps = (0, _utils.getValidSteps)(steps);
  return validSteps.length > 0;
};
const simulationMachine = exports.simulationMachine = (0, _xstate.setup)({
  types: {
    input: {},
    context: {},
    events: {}
  },
  actors: {
    runSimulation: (0, _xstateUtils.getPlaceholderFor)(_simulation_runner_actor.createSimulationRunnerActor)
  },
  actions: {
    notifySimulationRunFailure: (0, _xstateUtils.getPlaceholderFor)(_simulation_runner_actor.createSimulationRunFailureNotifier),
    storePreviewDocsFilter: (0, _xstate.assign)((_, params) => ({
      previewDocsFilter: params.filter
    })),
    storeSteps: (0, _xstate.assign)((_, params) => ({
      steps: params.steps
    })),
    storeSamples: (0, _xstate.assign)((_, params) => ({
      samples: params.samples
    })),
    storeSimulation: (0, _xstate.assign)((_, params) => ({
      simulation: params.simulation
    })),
    storeExplicitlyEnabledPreviewColumns: (0, _xstate.assign)(({
      context
    }, params) => ({
      explicitlyEnabledPreviewColumns: params.columns,
      explicitlyDisabledPreviewColumns: context.explicitlyDisabledPreviewColumns.filter(col => !params.columns.includes(col))
    })),
    storeExplicitlyDisabledPreviewColumns: (0, _xstate.assign)(({
      context
    }, params) => ({
      explicitlyDisabledPreviewColumns: params.columns,
      explicitlyEnabledPreviewColumns: context.explicitlyEnabledPreviewColumns.filter(col => !params.columns.includes(col))
    })),
    storePreviewColumnsOrder: (0, _xstate.assign)((_, params) => ({
      previewColumnsOrder: params.columns
    })),
    storePreviewColumnsSorting: (0, _xstate.assign)((_, params) => ({
      previewColumnsSorting: params.sorting
    })),
    deriveDetectedSchemaFields: (0, _xstate.assign)(({
      context
    }) => {
      const result = (0, _utils2.getSchemaFieldsFromSimulation)(context);
      return {
        detectedSchemaFields: result.detectedSchemaFields,
        detectedSchemaFieldsCache: result.detectedSchemaFieldsCache
      };
    }),
    mapField: (0, _xstate.assign)(({
      context
    }, params) => {
      const result = (0, _utils2.mapField)(context, params.field);
      return {
        detectedSchemaFields: result.detectedSchemaFields,
        detectedSchemaFieldsCache: result.detectedSchemaFieldsCache
      };
    }),
    unmapField: (0, _xstate.assign)(({
      context
    }, params) => {
      const result = (0, _utils2.unmapField)(context, params.fieldName);
      return {
        detectedSchemaFields: result.detectedSchemaFields,
        detectedSchemaFieldsCache: result.detectedSchemaFieldsCache
      };
    }),
    resetSimulationOutcome: (0, _xstate.assign)({
      detectedSchemaFields: [],
      explicitlyEnabledPreviewColumns: [],
      explicitlyDisabledPreviewColumns: [],
      previewColumnsOrder: [],
      simulation: undefined,
      previewDocsFilter: 'outcome_filter_all'
    }),
    resetSteps: (0, _xstate.assign)({
      steps: []
    }),
    resetSamples: (0, _xstate.assign)({
      samples: []
    })
  },
  delays: {
    processorChangeDebounceTime: 300
  },
  guards: {
    canSimulate: ({
      context
    }) => hasAnyValidSteps(context.steps),
    hasSteps: (_, params) => !(0, _lodash.isEmpty)(params.steps),
    '!hasSamples': (_, params) => !hasSamples(params.samples)
  }
}).createMachine({
  /** @xstate-layout N4IgpgJg5mDOIC5SwJYFsCuAbAhgFxQHsA7AYlU1wJIDoBjACx2JgAUAnMANxTAHcAIoTqwAYiix4w7ANoAGALqJQAB0KpqxZSAAeiAGwBmAJw1DAJgAs544cOWAHPoeGAjABoQAT0SvzAX39PCmx8IjIQqnCaTlgwPHklJBA1DXDtPQQAVgB2Tx8EV30cyxobfXNDEstLVztDQOD0UM1yZqjaTjowFC4wAH1YKRVYRO1UlE0MxFz8xGtXGgcsh2M5OVtXV0sjHMaQSLCSNsoj4hiwbt6B2Bw0FSw4MeSJqeTM2e9fStLcnP1XE4snZbA59odWhDol0en1BncHk9XElVOpJul3jM8l9soCzH5XHJlkS5NVwe0zicWtDLrCbgjHqNzCiUmi3qAPtiCtssqVLFVDMsbOZ9FkReTTq0VJwePwAMKELAYNDEWA0DAqCD4MAAUR0DxQdEmWC8OuIOAARo8IAqlSrRopxmyMRysXNCnUHDQ-DUtqT9DsVhLqcdpdxeHxbcrVerNdq9QajXgTQIULcrZAo-bnqi0iRptkub4csYcjQcnJzOYciZrHJXMYssGOmQw7LI4ro2rCOwINIc6y81pMYX3YTajRin45PZjI2qmCggcKVKZRGszG4ngAMo9ggsAevF26N04uq5GjGBy1HK31y5KyWZuUoZgFT0JgsMCH535kefApqz5ORciyLJCQqVwSybJcoWOV93z7R4pB-IcCwA4sy0bEUcgcKC5Esetn0hYYaCQ+Jv2RJ00P-ItCjAuQljWUkZxAup9GI6IUAgR4qRbGgADNeCwCA1TQHAVFQ9E-1dQp1kYqtRRrMDCKJMc7CyJYslLLJ9AIgFLBgppJS4niwD4s5BOE0T1WIcTJMdF5f2HWTCXWMoRVyQwVPWBwx35cx8XKQwKkMUkGlgldoj7C1CAwYgjRYOVPxgWBSB0IZtRoHABKkdgAAppWEOBYB7ZLmBgAQwFi+LugAFXQMAAEoLM0MjqrihKUCSlKnkc3NpJck85PcxSvJ8tScQsUpBUqPDgTkYor042gcFgOJ2H3KAACUwAARwwFBODQMBiDwNKpPZYatiKGh+RybT6wbMVLHdfQKhoGcHByPxymBa8VvONaNq23aDqOsATrOi6qKcmjXMWr1inMRbDPrRaPDPKxDEvRaAXMQF7BewGYni4huqgbcouOCASDAGhuq4QgAGt6bg4htrJ6QAEE6DwHtLuPTJLEbMw1gDXkKxKBwCPdb6cbnQE6hFCocnFSKTM6MmKapzWyGkdgexoB58AEns0BodnOeIYgeb5gX+sHQaCxFzSTFRyXSUcWWcVwxj-oDPDRSyesYKXYhCD7eBknZ6jnZHYx3TWO6xWcEOfRcSw9g1kNzm4x446uzI8MCwm3FLGWbF0sdqxxkOHBRytwIegGc-4mLOsSqByq-aOBqLxBcLdlGibAh71kMd0LBx2aq2MUVtjWCLjNz7L1ukUH9sO47TvOwuhd8CxFm0wUK35ZZCTo4FTH+DYjBR96HFWEn2G1lhddz-eZOuuoylyb6G7GDRhsSeOJRT6EvOxNWfhdIAjDv4IAA */
  id: 'simulation',
  context: ({
    input
  }) => ({
    detectedSchemaFields: [],
    detectedSchemaFieldsCache: new Map(),
    previewDocsFilter: 'outcome_filter_all',
    explicitlyDisabledPreviewColumns: [],
    explicitlyEnabledPreviewColumns: [],
    previewColumnsOrder: [],
    previewColumnsSorting: {
      fieldName: undefined,
      direction: 'asc'
    },
    steps: input.steps,
    samples: [],
    streamName: input.streamName,
    streamType: input.streamType
  }),
  initial: 'idle',
  on: {
    'simulation.changePreviewDocsFilter': {
      actions: [{
        type: 'storePreviewDocsFilter',
        params: ({
          event
        }) => event
      }]
    },
    'simulation.reset': {
      target: '.idle',
      actions: [{
        type: 'resetSimulationOutcome'
      }, {
        type: 'resetSteps'
      }]
    },
    'simulation.receive_steps': {
      target: '.assertingRequirements',
      actions: [{
        type: 'storeSteps',
        params: ({
          event
        }) => event
      }]
    },
    'simulation.receive_samples': [{
      guard: {
        type: '!hasSamples',
        params: ({
          event
        }) => event
      },
      target: '.idle',
      actions: [{
        type: 'resetSimulationOutcome'
      }, {
        type: 'resetSamples'
      }]
    }, {
      guard: {
        type: 'hasSteps',
        params: ({
          context
        }) => ({
          steps: context.steps
        })
      },
      target: '.assertingRequirements',
      actions: [{
        type: 'storeSamples',
        params: ({
          event
        }) => event
      }]
    }, {
      target: '.idle',
      actions: [{
        type: 'storeSamples',
        params: ({
          event
        }) => event
      }]
    }],
    'step.change': {
      target: '.debouncingChanges',
      reenter: true,
      description: 'Re-enter debouncing state and reinitialize the delayed processing.',
      actions: [{
        type: 'storeSteps',
        params: ({
          event
        }) => event
      }]
    },
    'step.delete': [{
      guard: {
        type: 'hasSteps',
        params: ({
          event
        }) => ({
          steps: event.steps
        })
      },
      target: '.assertingRequirements',
      actions: [{
        type: 'storeSteps',
        params: ({
          event
        }) => event
      }]
    }, {
      target: '.idle',
      actions: [{
        type: 'resetSimulationOutcome'
      }, {
        type: 'resetSteps'
      }]
    }]
  },
  states: {
    idle: {
      on: {
        'simulation.fields.map': {
          target: 'assertingRequirements',
          actions: [{
            type: 'mapField',
            params: ({
              event
            }) => event
          }]
        },
        'simulation.fields.unmap': {
          target: 'assertingRequirements',
          actions: [{
            type: 'unmapField',
            params: ({
              event
            }) => event
          }]
        },
        'previewColumns.updateExplicitlyEnabledColumns': {
          actions: [{
            type: 'storeExplicitlyEnabledPreviewColumns',
            params: ({
              event
            }) => event
          }]
        },
        'previewColumns.updateExplicitlyDisabledColumns': {
          actions: [{
            type: 'storeExplicitlyDisabledPreviewColumns',
            params: ({
              event
            }) => event
          }]
        },
        'previewColumns.order': {
          actions: [{
            type: 'storePreviewColumnsOrder',
            params: ({
              event
            }) => event
          }]
        },
        'previewColumns.setSorting': {
          actions: [{
            type: 'storePreviewColumnsSorting',
            params: ({
              event
            }) => event
          }]
        }
      }
    },
    debouncingChanges: {
      after: {
        processorChangeDebounceTime: 'assertingRequirements'
      }
    },
    assertingRequirements: {
      always: [{
        guard: 'canSimulate',
        target: 'runningSimulation'
      }, {
        target: 'idle',
        actions: [{
          type: 'resetSimulationOutcome'
        }]
      }]
    },
    runningSimulation: {
      invoke: {
        id: 'simulationRunnerActor',
        src: 'runSimulation',
        input: ({
          context
        }) => ({
          streamName: context.streamName,
          documents: context.samples.map(doc => doc.document).map(_objectUtils.flattenObjectNestedLast),
          steps: (0, _utils.getValidSteps)(context.steps),
          detectedFields: context.detectedSchemaFields
        }),
        onDone: {
          target: 'idle',
          actions: [{
            type: 'storeSimulation',
            params: ({
              event
            }) => ({
              simulation: event.output
            })
          }, {
            type: 'deriveDetectedSchemaFields'
          }]
        },
        onError: {
          target: 'idle',
          actions: [{
            type: 'notifySimulationRunFailure'
          }]
        }
      }
    }
  }
});
const createSimulationMachineImplementations = ({
  streamsRepositoryClient,
  toasts
}) => ({
  actors: {
    runSimulation: (0, _simulation_runner_actor.createSimulationRunnerActor)({
      streamsRepositoryClient
    })
  },
  actions: {
    notifySimulationRunFailure: (0, _simulation_runner_actor.createSimulationRunFailureNotifier)({
      toasts
    })
  }
});
exports.createSimulationMachineImplementations = createSimulationMachineImplementations;