"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDataSourceSelector = exports.dataSourceMachine = exports.createDataSourceMachineImplementations = void 0;
var _xstate = require("xstate5");
var _xstateUtils = require("@kbn/xstate-utils");
var _lodash = require("lodash");
var _react = require("@xstate5/react");
var _data_collector_actor = require("./data_collector_actor");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useDataSourceSelector = exports.useDataSourceSelector = _react.useSelector;
const dataSourceMachine = exports.dataSourceMachine = (0, _xstate.setup)({
  types: {
    input: {},
    context: {},
    events: {}
  },
  actors: {
    collectData: (0, _xstateUtils.getPlaceholderFor)(_data_collector_actor.createDataCollectorActor)
  },
  actions: {
    notifyDataCollectionFailure: (0, _xstateUtils.getPlaceholderFor)(_data_collector_actor.createDataCollectionFailureNotifier),
    restorePersistedCustomSamplesDocuments: (0, _xstate.assign)(({
      context
    }) => {
      if (context.dataSource.type === 'custom-samples' && context.dataSource.storageKey) {
        const dataSource = sessionStorage.getItem(context.dataSource.storageKey);
        if (dataSource) {
          return {
            dataSource: {
              ...context.dataSource,
              documents: JSON.parse(dataSource).documents
            }
          };
        }
      }
      return {};
    }),
    updatePersistedCustomSamplesDocuments: (0, _xstate.assign)(({
      context
    }) => {
      if (context.dataSource.type === 'custom-samples' && context.dataSource.storageKey) {
        sessionStorage.setItem(context.dataSource.storageKey, JSON.stringify(context.dataSource));
      }
      return {};
    }),
    removePersistedCustomSamplesDocuments: (0, _xstate.assign)(({
      context
    }) => {
      if (context.dataSource.type === 'custom-samples' && context.dataSource.storageKey) {
        sessionStorage.removeItem(context.dataSource.storageKey);
      }
      return {};
    }),
    storeDataSource: (0, _xstate.assign)(({
      context
    }, params) => ({
      dataSource: {
        ...params.dataSource,
        id: context.dataSource.id
      }
    })),
    storeData: (0, _xstate.assign)((_, params) => ({
      data: params.data
    })),
    toggleDataSourceActivity: (0, _xstate.assign)(({
      context
    }) => ({
      dataSource: {
        ...context.dataSource,
        enabled: !context.dataSource.enabled
      }
    })),
    notifyParent: (0, _xstate.sendTo)(({
      context
    }) => context.parentRef, ({
      context
    }, params) => ({
      type: params.eventType,
      id: context.dataSource.id
    }))
  },
  guards: {
    isEnabled: ({
      context
    }) => context.dataSource.enabled,
    isValidData: (_, params) => Array.isArray(params.data),
    shouldCollectData: ({
      context,
      event
    }) => {
      (0, _xstate.assertEvent)(event, 'dataSource.change');
      /**
       * Determines if the dataSource update contains substantive changes.
       * Ignores cosmetic changes like name updates that don't affect functionality.
       */
      const ignoredProps = ['name'];
      return !(0, _lodash.isEqual)((0, _lodash.omit)(context.dataSource, ignoredProps), (0, _lodash.omit)(event.dataSource, ignoredProps));
    }
  }
}).createMachine({
  /** @xstate-layout N4IgpgJg5mDOIC5QQIYBcUGUD2BXATgMZgB0EYaY+AtgJYB2DUAxANoAMAuoqAA7axaaWtno8QAD0QBaAIwAmEu1kAOAMzsAbLIDsAVnX75agDQgAnogM6SagJzLNmlXYAsRtQF9PZ1BhwExGQUVHSM9CysstxIIPyCwqLiUgiyziSyrirZdpp2eqp6Oq5mlggGmiQ6dvJ6blqaOuzy3r7oWHhEpGD0KABGADaQzH4dgaRo2FBQQwCChMIAbkLmHDF8AkIiYrEpdZWqafI6asdqeux6pYia7iTOddWushfydq0gowFdJD39QxARu1vkF8GAAGZg2AACzW4niWySu0Q+SUdh0TVc2hO7DsL2uCBcdhI+wK52Mb3ksg+X06QT+g2GtPGJEI0JQETAcNiCMSO1AKSxrgyeQcKiKJwuVwsKKaSip7Cyenkmj0zlcNOBdO6vUZgOZPzZHJgUXWcU2fOSiBcKhJpzU4tkdnUdWlZXR7HlskV4pVavFmv82t+uoBJAG2BQECYABF2swqPhsPgSLwBuhwcnqGR2gBhbADIYLZPzSb4bkbBLbK0IHS2716PRZfJUpwYgne1FNzs6PIaWReHyfLUs6Owf5Mkc-SbTOYLWjLNCrLjwi3V5EIDRqEmZPLPE54tQlGUIW42B7otwvZrvIcGoJjif6qcPsBDSgV81VpECqyabf5E8KhYgUFyyASBSKLkOjyG8zyvLeQ70Ng5DwLE95gKu378pIMhOCQjquBKbzKmovYEs8nryM8ThFG8djnERgZjD85CUDQDBMFhiI4Sk0h6CQrjsC4W6nLI4nKPIEGKrYW6NLUInHMxII6k+3GWhuci2oRxEtmRmgEioChorcmjsGoLwuHoynBgyYa0BAQzqeuv6pMqsl2G4REYmomjHB28jAfcVKXk4zSqjoNksnZkDhpG0YRHGGDOT+uEIG8GRKq4sGYr21EEs6UHFLo5KquqUWsbQ456ilvGIL525UoU7gvNemQQeKmXKER7C9ioVKDm0Qajm+IQQLVNbATY2V5GkrhZOcuQUeiVQ0b6lwuK4iGeEAA */
  id: 'dataSource',
  context: ({
    input
  }) => ({
    parentRef: input.parentRef,
    data: [],
    dataSource: input.dataSource,
    streamName: input.streamName,
    simulationMode: getSimulationModeByDataSourceType(input.dataSource.type)
  }),
  initial: 'determining',
  states: {
    determining: {
      entry: [{
        type: 'restorePersistedCustomSamplesDocuments'
      }],
      always: [{
        target: 'enabled',
        guard: 'isEnabled'
      }, {
        target: 'disabled'
      }]
    },
    enabled: {
      initial: 'loadingData',
      on: {
        'dataSource.disable': {
          target: 'disabled',
          actions: [{
            type: 'toggleDataSourceActivity'
          }, {
            type: 'updatePersistedCustomSamplesDocuments'
          }, {
            type: 'notifyParent',
            params: {
              eventType: 'dataSource.change'
            }
          }]
        },
        'dataSource.refresh': {
          target: '.loadingData',
          reenter: true
        },
        'dataSource.change': [{
          guard: 'shouldCollectData',
          target: '.loadingData',
          reenter: true,
          actions: [{
            type: 'storeDataSource',
            params: ({
              event
            }) => event
          }, {
            type: 'notifyParent',
            params: {
              eventType: 'dataSource.change'
            }
          }]
        }, {
          actions: [{
            type: 'storeDataSource',
            params: ({
              event
            }) => event
          }, {
            type: 'notifyParent',
            params: {
              eventType: 'dataSource.change'
            }
          }]
        }]
      },
      states: {
        idle: {},
        loadingData: {
          invoke: {
            id: 'dataCollectorActor',
            src: 'collectData',
            input: ({
              context
            }) => ({
              dataSource: context.dataSource,
              streamName: context.streamName
            }),
            onSnapshot: {
              guard: {
                type: 'isValidData',
                params: ({
                  event
                }) => ({
                  data: event.snapshot.context
                })
              },
              target: 'idle',
              actions: [{
                type: 'storeData',
                params: ({
                  event
                }) => {
                  var _event$snapshot$conte;
                  return {
                    data: (_event$snapshot$conte = event.snapshot.context) !== null && _event$snapshot$conte !== void 0 ? _event$snapshot$conte : []
                  };
                }
              }, {
                type: 'notifyParent',
                params: {
                  eventType: 'dataSource.dataChange'
                }
              }]
            },
            onError: {
              target: 'idle',
              actions: [{
                type: 'storeData',
                params: () => ({
                  data: []
                })
              }, {
                type: 'notifyParent',
                params: {
                  eventType: 'dataSource.dataChange'
                }
              }, {
                type: 'notifyDataCollectionFailure'
              }]
            }
          }
        }
      }
    },
    disabled: {
      on: {
        'dataSource.enable': {
          target: 'enabled',
          actions: [{
            type: 'toggleDataSourceActivity'
          }, {
            type: 'updatePersistedCustomSamplesDocuments'
          }, {
            type: 'notifyParent',
            params: {
              eventType: 'dataSource.change'
            }
          }]
        },
        'dataSource.change': {
          actions: [{
            type: 'storeDataSource',
            params: ({
              event
            }) => event
          }, {
            type: 'notifyParent',
            params: {
              eventType: 'dataSource.change'
            }
          }]
        },
        'dataSource.delete': 'deleted'
      }
    },
    deleted: {
      id: 'deleted',
      type: 'final',
      entry: [{
        type: 'notifyParent',
        params: {
          eventType: 'dataSource.delete'
        }
      }, {
        type: 'removePersistedCustomSamplesDocuments'
      }]
    }
  }
});
const createDataSourceMachineImplementations = ({
  data,
  toasts
}) => ({
  actors: {
    collectData: (0, _data_collector_actor.createDataCollectorActor)({
      data
    })
  },
  actions: {
    notifyDataCollectionFailure: (0, _data_collector_actor.createDataCollectionFailureNotifier)({
      toasts
    })
  }
});
exports.createDataSourceMachineImplementations = createDataSourceMachineImplementations;
const getSimulationModeByDataSourceType = dataSourceType => {
  switch (dataSourceType) {
    case 'latest-samples':
      return 'partial';
    case 'kql-samples':
      return 'partial';
    case 'custom-samples':
      return 'complete';
    default:
      throw new Error(`Invalid data source type: ${dataSourceType}`);
  }
};