"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.identifySystemFeatures = identifySystemFeatures;
var _aiTools = require("@kbn/ai-tools");
var _inferencePromptUtils = require("@kbn/inference-prompt-utils");
var _streamsSchema = require("@kbn/streams-schema");
var _prompt = require("./prompt");
var _cluster_logs = require("../cluster_logs/cluster_logs");
var _condition_schema = _interopRequireDefault(require("../shared/condition_schema.text"));
var _generate_description = require("../description/generate_description");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Identifies features in a stream, by:
 * - describing the dataset (via sampled documents)
 * - clustering docs together on similarity
 * - asking the LLM to identify features by creating
 * queries and validating the resulting clusters
 */
async function identifySystemFeatures({
  stream,
  features,
  start,
  end,
  esClient,
  inferenceClient,
  logger,
  signal,
  analysis,
  dropUnmapped = false,
  maxSteps: initialMaxSteps
}) {
  var _features$filter$map;
  const initialClustering = await (0, _cluster_logs.clusterLogs)({
    start,
    end,
    esClient,
    index: stream.name,
    partitions: (_features$filter$map = features === null || features === void 0 ? void 0 : features.filter(_streamsSchema.isFeatureWithFilter).map(feature => {
      return {
        name: feature.name,
        condition: feature.filter
      };
    })) !== null && _features$filter$map !== void 0 ? _features$filter$map : [],
    logger,
    dropUnmapped
  });
  const response = await (0, _inferencePromptUtils.executeAsReasoningAgent)({
    maxSteps: initialMaxSteps,
    input: {
      stream: {
        name: stream.name,
        description: stream.description || 'This stream has no description.'
      },
      dataset_analysis: JSON.stringify((0, _aiTools.formatDocumentAnalysis)(analysis, {
        dropEmpty: true,
        dropUnmapped
      })),
      initial_clustering: JSON.stringify(initialClustering),
      condition_schema: _condition_schema.default
    },
    prompt: _prompt.IdentifySystemsPrompt,
    inferenceClient,
    finalToolChoice: {
      function: 'finalize_systems'
    },
    toolCallbacks: {
      validate_systems: async toolCall => {
        const clustering = await (0, _cluster_logs.clusterLogs)({
          start,
          end,
          esClient,
          index: stream.name,
          logger,
          partitions: toolCall.function.arguments.systems.map(system => {
            return {
              name: system.name,
              condition: system.filter
            };
          }),
          dropUnmapped
        });
        return {
          response: {
            systems: clustering.map(cluster => {
              return {
                name: cluster.name,
                clustering: cluster.clustering
              };
            })
          }
        };
      },
      finalize_systems: async toolCall => {
        return {
          response: {}
        };
      }
    },
    abortSignal: signal
  });
  return {
    features: await Promise.all(response.toolCalls.flatMap(toolCall => toolCall.function.arguments.systems.map(args => {
      const feature = {
        ...args,
        filter: args.filter,
        type: 'system'
      };
      return feature;
    })).map(async feature => {
      const description = await (0, _generate_description.generateStreamDescription)({
        stream,
        start,
        end,
        esClient,
        inferenceClient,
        feature: {
          ...feature,
          description: ''
        },
        signal
      });
      return {
        ...feature,
        description
      };
    }))
  };
}