"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StreamlangTargetField = exports.StreamlangSourceField = exports.StreamlangSeparator = exports.NoMustacheValue = exports.NoMustacheArrayValues = void 0;
var _zod = require("@kbn/zod");
var _zodHelpers = require("@kbn/zod-helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * String schema that rejects Mustache template syntax.
 * Validation at Zod level ensures immediate feedback in Streamlang editors,
 * validation APIs, and prevents invalid DSL creation entirely.
 * Checks for `{{` anywhere in the string which makes Mustache active in Ingest Pipelines.
 */
const NoMustacheString = _zodHelpers.NonEmptyString.refine(val => !val.includes('{{'), {
  message: 'Mustache template syntax {{ }} or {{{ }}} is not allowed in field names'
});

/**
 * Smart validation that rejects Mustache templates only if the value is a string.
 * Non-string values (numbers, booleans, etc.) pass through since they can't contain templates.
 */
const NoMustacheValue = exports.NoMustacheValue = _zod.z.unknown().refine(val => {
  if (typeof val === 'string') {
    return !val.includes('{{');
  }
  return true; // Non-strings are allowed
}, {
  message: 'Mustache template syntax {{ }} or {{{ }}} is not allowed in [value] field'
});

/**
 * Array validation that rejects Mustache templates in any string elements.
 * Non-string array elements (numbers, booleans, etc.) pass through.
 */
const NoMustacheArrayValues = exports.NoMustacheArrayValues = _zod.z.array(_zod.z.unknown()).nonempty().refine(arr => {
  return arr.every(val => {
    if (typeof val === 'string') {
      return !val.includes('{{');
    }
    return true; // Non-strings are allowed
  });
}, {
  message: 'Mustache template syntax {{ }} or {{{ }}} is not allowed in array [value] elements'
});

/**
 * Source field names (from fields) - prevents Mustache templates in field references
 */
const StreamlangSourceField = exports.StreamlangSourceField = NoMustacheString;

/**
 * Target field names (to fields) - prevents Mustache templates in field creation
 */
const StreamlangTargetField = exports.StreamlangTargetField = NoMustacheString;

/**
 * Separator for join and split processors. Allows spaces and other characters.
 */
const StreamlangSeparator = exports.StreamlangSeparator = _zod.z.string().min(1);