"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertRemoveProcessorToESQL = convertRemoveProcessorToESQL;
var _esqlAst = require("@kbn/esql-ast");
var _common = require("./common");
var _condition_to_esql = require("../condition_to_esql");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts a Streamlang RemoveProcessor into a list of ES|QL AST commands.
 *
 * For unconditional removal (no 'where' or 'where: always'):
 *   Uses DROP command to remove the field
 *
 * For conditional removal (with 'where' condition):
 *   Uses EVAL with CASE to set field to null when condition matches:
 *   EVAL field = CASE(<condition>, null, field)
 *
 * Filters applied for Ingest Pipeline parity:
 * - When `ignore_missing: false`: `WHERE NOT(field IS NULL)` filters missing fields
 *
 * @example Unconditional:
 *    ```typescript
 *    const streamlangDSL: StreamlangDSL = {
 *      steps: [
 *        {
 *          action: 'remove',
 *          from: 'temp_field',
 *        } as RemoveProcessor,
 *      ],
 *    };
 *    ```
 *
 *    Generates:
 *    ```txt
 *    | DROP temp_field
 *    ```
 *
 * @example Conditional:
 *    ```typescript
 *    const streamlangDSL: StreamlangDSL = {
 *      steps: [
 *        {
 *          action: 'remove',
 *          from: 'temp_field',
 *          where: { field: 'status', eq: 'test' },
 *        } as RemoveProcessor,
 *      ],
 *    };
 *    ```
 *
 *    Generates:
 *    ```txt
 *    | EVAL temp_field = CASE(status == "test", null, temp_field)
 *    ```
 */
function convertRemoveProcessorToESQL(processor) {
  const {
    from,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    ignore_missing = false // default: false (field must exist)
  } = processor;
  const commands = [];

  // Check if this is conditional or unconditional removal

  if ('where' in processor && processor.where && !('always' in processor.where)) {
    // Conditional removal: use EVAL with CASE to null the field when condition matches
    // EVAL field = CASE(<condition>, null, field)
    const conditionExpression = (0, _condition_to_esql.conditionToESQLAst)(processor.where);
    const caseExpression = _esqlAst.Builder.expression.func.call('CASE', [conditionExpression, _esqlAst.Builder.expression.literal.nil(), _esqlAst.Builder.expression.column(from)]);
    const evalCommand = _esqlAst.Builder.command({
      name: 'eval',
      args: [_esqlAst.Builder.expression.func.binary('=', [_esqlAst.Builder.expression.column(from), caseExpression])]
    });
    commands.push(evalCommand);
  } else {
    // Unconditional removal: use DROP command
    // Add missing field filter if needed (ignore_missing = false)
    const missingFieldFilter = (0, _common.buildIgnoreMissingFilter)(from, ignore_missing);
    if (missingFieldFilter) {
      commands.push(missingFieldFilter);
    }
    const dropCommand = _esqlAst.Builder.command({
      name: 'drop',
      args: [_esqlAst.Builder.expression.column(from)]
    });
    commands.push(dropCommand);
  }
  return commands;
}