"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertGrokProcessorToESQL = convertGrokProcessorToESQL;
var _esqlAst = require("@kbn/esql-ast");
var _grok_patterns = require("../../../../types/utils/grok_patterns");
var _condition_to_esql = require("../condition_to_esql");
var _common = require("./common");
var _grok_pattern_definitions = require("../../../../types/utils/grok_pattern_definitions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts a Streamlang GrokProcessor into a list of ES|QL AST commands.
 *
 * Conditional execution logic:
 *  - If neither `ignore_missing` nor `where` is provided: emit a single GROK command.
 *  - Otherwise, use CASE approach to conditionally execute GROK:
 *      * Create temporary field and use CASE to conditionally set source field or empty string (empty string avoids ES|QL NULL errors)
 *      * Apply GROK to temporary field
 *      * Drop temporary field
 *    Condition: (exists(from) if ignore_missing) AND (where condition, if provided)
 *
 * Type handling:
 *  - Pre-grok: cast all GROKed target fields to their suffixed (or default) types with
 *              TO_STRING (keyword), TO_INTEGER or TO_DOUBLE for consistency.
 *
 * @example:
 *    ```typescript
 *    const streamlangDSL: StreamlangDSL = {
 *      steps: [
 *        {
 *          action: 'grok',
 *          from: 'message',
 *          patterns: ["%{IP:client.ip} %{NUMBER:size:int} %{NUMBER:burn_rate:float}"],
 *          ignore_missing: true,
 *          where: { field: 'flags.process', exists: true },
 *        } as GrokProcessor,
 *      ],
 *    };
 *    ```
 *
 *    Generates (conceptually):
 *    ```txt
 *    // | WHERE NOT(message IS NULL)  // Only if ignore_missing = false
 *    | EVAL `client.ip` = TO_STRING(`client.ip`)
 *    | EVAL `size` = TO_INTEGER(`size`)
 *    | EVAL `burn_rate` = TO_DOUBLE(`burn_rate`)
 *    | EVAL __temp_grok_where_message__ = CASE(NOT(message IS NULL) AND NOT(`flags.process` IS NULL), message, "")
 *    | GROK __temp_grok_where_message__ "%{IP:client.ip} %{NUMBER:size:int} %{NUMBER:burn_rate:float}"
 *    | DROP __temp_grok_where_message__
 *    ```
 */
function convertGrokProcessorToESQL(processor) {
  const {
    from,
    // eslint-disable-next-line @typescript-eslint/naming-convention
    ignore_missing = false,
    // default mirrors ingest grok behavior
    where
  } = processor;
  const fromColumn = _esqlAst.Builder.expression.column(from);
  const primaryPattern = (0, _grok_pattern_definitions.unwrapPatternDefinitions)(processor)[0];
  const grokCommand = buildGrokCommand(fromColumn, primaryPattern);
  const commands = [];

  // Add missing field filter if needed (ignore_missing = false)
  const missingFieldFilter = (0, _common.buildIgnoreMissingFilter)(from, ignore_missing);
  if (missingFieldFilter) {
    commands.push(missingFieldFilter);
  }

  // Check if conditional execution is needed for 'where' clauses, return simple command otherwise
  const needConditional = ignore_missing || Boolean(where);
  if (!needConditional) {
    commands.push(grokCommand);
    return commands;
  }

  // Pre-cast existing target fields to their configured GROK types to avoid ES|QL type conflict errors
  const {
    allFields
  } = (0, _grok_patterns.parseMultiGrokPatterns)([primaryPattern]);
  if (allFields.length > 0) {
    commands.push(...(0, _common.castFieldsToGrokTypes)(allFields));
  }

  // Build condition for when GROK should execute
  const grokCondition = (0, _common.buildWhereCondition)(from, ignore_missing, where, _condition_to_esql.conditionToESQLAst);

  // Create temporary field name for conditional processing
  // Using CASE, set temporary field to source field if condition passes, empty string otherwise
  const tempFieldName = `__temp_grok_where_${from}__`;
  const tempColumn = _esqlAst.Builder.expression.column(tempFieldName);

  //
  commands.push(_esqlAst.Builder.command({
    name: 'eval',
    args: [_esqlAst.Builder.expression.func.binary('=', [tempColumn, _esqlAst.Builder.expression.func.call('CASE', [grokCondition, fromColumn, _esqlAst.Builder.expression.literal.string('') // Empty string avoids ES|QL NULL errors that would break the pipeline
    ])])]
  }));

  // Apply GROK to the temporary field
  commands.push(buildGrokCommand(tempColumn, primaryPattern));

  // Clean up temporary field
  commands.push(_esqlAst.Builder.command({
    name: 'drop',
    args: [tempColumn]
  }));
  return commands;
}

/** Build the GROK command (primary pattern only) */
function buildGrokCommand(fromColumn, pattern) {
  return _esqlAst.Builder.command({
    name: 'grok',
    args: [fromColumn, _esqlAst.Builder.expression.literal.string(pattern)]
  });
}