"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getBinaryFilterField = getBinaryFilterField;
exports.getBinaryFilterOperator = getBinaryFilterOperator;
exports.getBinaryFilterValue = getBinaryFilterValue;
exports.getConditionFields = getConditionFields;
exports.getDefaultFormValueForOperator = getDefaultFormValueForOperator;
exports.getFilterOperator = getFilterOperator;
exports.getFilterValue = getFilterValue;
exports.getUnaryFilterField = getUnaryFilterField;
exports.getUnaryFilterOperator = getUnaryFilterOperator;
exports.getUnaryFilterValue = getUnaryFilterValue;
exports.isFilterConditionObject = isFilterConditionObject;
var _conditions = require("../../types/conditions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getBinaryFilterOperator(condition) {
  return _conditions.BINARY_OPERATORS.find(op => condition[op] !== undefined);
}
function getBinaryFilterValue(condition) {
  const operator = getBinaryFilterOperator(condition);
  return operator ? condition[operator] : undefined;
}
function getBinaryFilterField(condition) {
  return condition.field;
}
function getUnaryFilterOperator(condition) {
  return _conditions.UNARY_OPERATORS.find(op => condition[op] !== undefined);
}
function getUnaryFilterValue(condition) {
  const operator = getUnaryFilterOperator(condition);
  return operator ? condition[operator] : undefined;
}
function getUnaryFilterField(condition) {
  return condition.field;
}
function getFilterValue(condition) {
  const binaryOp = getBinaryFilterOperator(condition);
  if (binaryOp) {
    return condition[binaryOp];
  }
  const unaryOp = getUnaryFilterOperator(condition);
  if (unaryOp) {
    return condition[unaryOp];
  }
  return undefined;
}
function getFilterOperator(condition) {
  const binaryOp = getBinaryFilterOperator(condition);
  if (binaryOp) {
    return binaryOp;
  }
  const unaryOp = getUnaryFilterOperator(condition);
  if (unaryOp) {
    return unaryOp;
  }
  return undefined;
}
function getDefaultFormValueForOperator(operator) {
  switch (operator) {
    case 'eq':
    case 'neq':
    case 'contains':
    case 'startsWith':
    case 'endsWith':
    case 'lt':
    case 'lte':
    case 'gt':
    case 'gte':
      return '';
    case 'exists':
      return true;
    case 'range':
      return {
        gte: 0,
        lt: 0
      };
    default:
      return '';
  }
}
function isFilterConditionObject(condition) {
  const allOperators = [..._conditions.BINARY_OPERATORS, ..._conditions.UNARY_OPERATORS];
  return allOperators.some(op => op in condition);
}
function getConditionFields(condition) {
  const fields = collectFields(condition);
  // deduplicate fields, if already mapped, prefer boolean over number, and number over string
  const uniqueFields = new Map();
  const typePrecedenceAscOrder = ['string', 'number', 'boolean'];
  fields.forEach(({
    name,
    type
  }) => {
    const existing = uniqueFields.get(name);
    if (!existing || typePrecedenceAscOrder.indexOf(type) > typePrecedenceAscOrder.indexOf(existing)) {
      uniqueFields.set(name, type);
    }
  });
  return Array.from(uniqueFields).map(([name, type]) => ({
    name,
    type
  }));
}
function collectFields(condition) {
  if ((0, _conditions.isFilterCondition)(condition)) {
    return [{
      name: condition.field,
      type: getFieldTypeForFilterCondition(condition)
    }];
  }
  if ((0, _conditions.isAndCondition)(condition)) {
    return condition.and.flatMap(collectFields);
  }
  if ((0, _conditions.isOrCondition)(condition)) {
    return condition.or.flatMap(collectFields);
  }
  if ((0, _conditions.isNotCondition)(condition)) {
    return collectFields(condition.not);
  }
  return [];
}
function getFieldTypeForFilterCondition(condition) {
  const operator = getFilterOperator(condition);
  const value = getFilterValue(condition);
  switch (operator) {
    case 'gt':
    case 'gte':
    case 'lt':
    case 'lte':
      return 'number';
    case 'neq':
    case 'eq':
      // Return number or boolean if the value is of that type, otherwise string
      if (typeof value === 'number') {
        return 'number';
      }
      if (typeof value === 'boolean') {
        return 'boolean';
      }
      return 'string';
    case 'exists':
    case 'contains':
    case 'startsWith':
    case 'endsWith':
      return 'string';
    default:
      return 'string';
  }
}