"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executor = executor;
exports.getConnectorType = getConnectorType;
var _pipeable = require("fp-ts/pipeable");
var _Option = require("fp-ts/Option");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _common = require("@kbn/actions-plugin/common");
var _mustache_renderer = require("@kbn/actions-plugin/server/lib/mustache_renderer");
var _common2 = require("@kbn/task-manager-plugin/common");
var _auth = require("@kbn/connector-schemas/common/auth");
var _webhook = require("@kbn/connector-schemas/webhook");
var _http_response_retry_header = require("../lib/http_response_retry_header");
var _result_type = require("../lib/result_type");
var _get_axios_config = require("./get_axios_config");
var _validations = require("./validations");
var _errors = require("./errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const userErrorCodes = [400, 404, 405, 406, 410, 411, 414, 428, 431];

// connector type definition
function getConnectorType() {
  return {
    id: _webhook.CONNECTOR_ID,
    minimumLicenseRequired: 'gold',
    name: _webhook.CONNECTOR_NAME,
    supportedFeatureIds: [_common.AlertingConnectorFeatureId, _common.UptimeConnectorFeatureId, _common.SecurityConnectorFeatureId],
    validate: {
      config: {
        schema: _webhook.ConfigSchema,
        customValidator: _validations.validateConnectorTypeConfig
      },
      secrets: {
        schema: _auth.SecretConfigurationSchema
      },
      params: {
        schema: _webhook.ParamsSchema
      }
    },
    renderParameterTemplates,
    executor
  };
}
function renderParameterTemplates(logger, params, variables) {
  if (!params.body) return params;
  return {
    body: (0, _mustache_renderer.renderMustacheString)(logger, params.body, variables, 'json')
  };
}
function methodExpectsBody({
  method
}) {
  return ![_auth.WebhookMethods.GET, _auth.WebhookMethods.DELETE].includes(method);
}

// action executor
async function executor(execOptions) {
  const {
    actionId,
    config,
    params,
    configurationUtilities,
    logger,
    connectorUsageCollector,
    services
  } = execOptions;
  const {
    method,
    url
  } = config;
  const {
    body: data
  } = params;
  const [axiosConfig, axiosConfigError] = await (0, _get_axios_config.getAxiosConfig)({
    connectorId: actionId,
    services,
    config,
    secrets: execOptions.secrets,
    configurationUtilities,
    logger
  });
  if (axiosConfigError) {
    var _axiosConfigError$mes, _axiosConfigError$mes2;
    logger.error(`ConnectorId "${actionId}": error "${(_axiosConfigError$mes = axiosConfigError.message) !== null && _axiosConfigError$mes !== void 0 ? _axiosConfigError$mes : 'unknown error - couldnt load axios config'}"`);
    return (0, _errors.errorResultRequestFailed)(actionId, (_axiosConfigError$mes2 = axiosConfigError.message) !== null && _axiosConfigError$mes2 !== void 0 ? _axiosConfigError$mes2 : 'unknown error - couldnt load axios config');
  }
  const {
    axiosInstance,
    headers,
    sslOverrides
  } = axiosConfig;
  const result = await (0, _result_type.promiseResult)((0, _axios_utils.request)({
    axios: axiosInstance,
    method,
    url,
    logger,
    headers,
    data: methodExpectsBody({
      method
    }) ? data : undefined,
    configurationUtilities,
    sslOverrides,
    connectorUsageCollector
  }));
  if (result == null) {
    return (0, _errors.errorResultUnexpectedNullResponse)(actionId);
  }
  if ((0, _result_type.isOk)(result)) {
    const {
      value: {
        status,
        statusText
      }
    } = result;
    logger.debug(`response from webhook action "${actionId}": [HTTP ${status}] ${statusText}`);
    return successResult(actionId, data);
  } else {
    const {
      error
    } = result;
    if (error.response) {
      const {
        status,
        statusText,
        headers: responseHeaders,
        data: {
          message: responseMessage
        }
      } = error.response;
      const responseMessageAsSuffix = responseMessage ? `: ${responseMessage}` : '';
      const message = `[${status}] ${statusText}${responseMessageAsSuffix}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      // The request was made and the server responded with a status code
      // that falls out of the range of 2xx
      // special handling for 5xx
      if (status >= 500) {
        return (0, _errors.retryResult)(actionId, message);
      }

      // special handling for rate limiting
      if (status === 429) {
        return (0, _pipeable.pipe)((0, _http_response_retry_header.getRetryAfterIntervalFromHeaders)(responseHeaders), (0, _Option.map)(retry => (0, _errors.retryResultSeconds)(actionId, message, retry)), (0, _Option.getOrElse)(() => (0, _errors.retryResult)(actionId, message)));
      }
      const errorResult = (0, _errors.errorResultInvalid)(actionId, message);
      if (userErrorCodes.includes(status)) {
        errorResult.errorSource = _common2.TaskErrorSource.USER;
      }
      return errorResult;
    } else if (error.code) {
      const message = `[${error.code}] ${error.message}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      return (0, _errors.errorResultRequestFailed)(actionId, message);
    } else if (error.isAxiosError) {
      const message = `${error.message}`;
      logger.error(`error on ${actionId} webhook event: ${message}`);
      return (0, _errors.errorResultRequestFailed)(actionId, message);
    }
    logger.error(`error on ${actionId} webhook action: unexpected error`);
    return (0, _errors.errorResultUnexpectedError)(actionId);
  }
}

// Action Executor Result w/ internationalisation
function successResult(actionId, data) {
  return {
    status: 'ok',
    data,
    actionId
  };
}