"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createExternalService = void 0;
var _axios = _interopRequireDefault(require("axios"));
var _i18n = require("@kbn/i18n");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _pipeable = require("fp-ts/pipeable");
var _Option = require("fp-ts/Option");
var _task_running = require("@kbn/task-manager-plugin/server/task_running");
var _slack_api = require("@kbn/connector-schemas/slack_api");
var _lib = require("../../../common/slack_api/lib");
var _http_response_retry_header = require("../lib/http_response_retry_header");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildSlackExecutorErrorResponse = ({
  slackApiError,
  logger
}) => {
  if (!slackApiError.response) {
    return (0, _lib.serviceErrorResult)(_slack_api.CONNECTOR_ID, slackApiError.message);
  }
  const {
    status,
    statusText,
    headers
  } = slackApiError.response;

  // special handling for 5xx
  if (status >= 500) {
    return (0, _lib.retryResult)(_slack_api.CONNECTOR_ID, slackApiError.message);
  }

  // special handling for rate limiting
  if (status === 429) {
    return (0, _pipeable.pipe)((0, _http_response_retry_header.getRetryAfterIntervalFromHeaders)(headers), (0, _Option.map)(retry => (0, _lib.retryResultSeconds)(_slack_api.CONNECTOR_ID, slackApiError.message, retry)), (0, _Option.getOrElse)(() => (0, _lib.retryResult)(_slack_api.CONNECTOR_ID, slackApiError.message)));
  }
  const errorMessage = _i18n.i18n.translate('xpack.stackConnectors.slack.unexpectedHttpResponseErrorMessage', {
    defaultMessage: 'unexpected http response from slack: {httpStatus} {httpStatusText}',
    values: {
      httpStatus: status,
      httpStatusText: statusText
    }
  });
  logger.error(`error on ${_slack_api.CONNECTOR_ID} slack action: ${errorMessage}`);
  const errorSource = (0, _task_running.getErrorSource)(slackApiError);
  return (0, _lib.errorResult)(_slack_api.CONNECTOR_ID, errorMessage, errorSource);
};
const buildSlackExecutorSuccessResponse = ({
  slackApiResponseData
}) => {
  if (!slackApiResponseData) {
    const errMessage = _i18n.i18n.translate('xpack.stackConnectors.slack.unexpectedNullResponseErrorMessage', {
      defaultMessage: 'unexpected null response from slack'
    });
    return (0, _lib.errorResult)(_slack_api.CONNECTOR_ID, errMessage);
  }
  if (!slackApiResponseData.ok) {
    return (0, _lib.serviceErrorResult)(_slack_api.CONNECTOR_ID, slackApiResponseData.error);
  }
  return (0, _lib.successResult)(_slack_api.CONNECTOR_ID, slackApiResponseData);
};
const createExternalService = ({
  config,
  secrets
}, logger, configurationUtilities, connectorUsageCollector) => {
  const {
    token
  } = secrets;
  const {
    allowedChannels
  } = config || {
    allowedChannels: []
  };
  const allowedChannelIds = allowedChannels === null || allowedChannels === void 0 ? void 0 : allowedChannels.map(ac => ac.id).filter(id => id !== undefined);
  const allowedChannelNames = allowedChannels === null || allowedChannels === void 0 ? void 0 : allowedChannels.map(ac => ac.name);
  if (!token) {
    throw Error(`[Action][${_slack_api.CONNECTOR_NAME}]: Wrong configuration.`);
  }
  const axiosInstance = _axios.default.create();
  const headers = {
    Authorization: `Bearer ${token}`,
    'Content-type': 'application/json; charset=UTF-8'
  };
  const validChannelId = async channelId => {
    try {
      const validChannel = () => {
        return (0, _axios_utils.request)({
          axios: axiosInstance,
          configurationUtilities,
          logger,
          method: 'get',
          headers,
          url: `${_slack_api.SLACK_URL}conversations.info?channel=${channelId}`,
          connectorUsageCollector
        });
      };
      if (channelId.length === 0) {
        return buildSlackExecutorErrorResponse({
          slackApiError: new Error('The channel id is empty'),
          logger
        });
      }
      const result = await validChannel();
      return buildSlackExecutorSuccessResponse({
        slackApiResponseData: result.data
      });
    } catch (error) {
      return buildSlackExecutorErrorResponse({
        slackApiError: error,
        logger
      });
    }
  };
  const validateChannels = ({
    channels,
    allowedList
  }) => {
    if (!channels || !channels.length || !allowedList || !allowedList.length) return;
    const normalizeChannel = name => name.replace(/^#/, '');
    const hasDisallowedChannel = channels === null || channels === void 0 ? void 0 : channels.some(name => !allowedList.some(allowed => normalizeChannel(allowed) === normalizeChannel(name)));
    if (hasDisallowedChannel) {
      throw new Error(`One or more provided channel names are not included in the allowed channels list`);
    }
  };

  /**
   * Selects the Slack channel to use for message delivery. At the moment, only posting to a single channel is supported.
   *
   * Priority order:
   *   1. If channelNames is provided and non-empty, validates against allowedChannelNames (if configured) and returns the first entry.
   *   2. If channelIds is provided and non-empty, validates against allowedChannelIds (if configured) and returns the first entry.
   *   3. If channels (legacy) is provided and non-empty, returns the first entry.
   *   4. Throws if none are provided or all are empty.
   *
   * If allowedChannels is empty or undefined, no validation is performed against allowedChannelNames or allowedChannelIds.
   */
  const getChannelToUse = ({
    channels = [],
    channelIds = [],
    channelNames = []
  }) => {
    const hasChannels = channelNames.length > 0 || channelIds.length > 0 || channels.length > 0;
    if (!hasChannels) {
      throw new Error(`One of channels, channelIds, or channelNames is required and cannot be empty`);
    }

    // priority: channelNames > channelIds > channels
    if (channelNames.length > 0) {
      validateChannels({
        channels: channelNames,
        allowedList: allowedChannelNames
      });
      return channelNames[0];
    }
    if (channelIds.length > 0) {
      validateChannels({
        channels: channelIds,
        allowedList: allowedChannelIds
      });
      return channelIds[0];
    }
    if (channels && channels.length > 0) {
      return channels[0];
    }
    throw new Error(`No valid channel found to use`);
  };
  const postMessage = async ({
    channels = [],
    channelIds = [],
    channelNames = [],
    text
  }) => {
    try {
      const channelToUse = getChannelToUse({
        channels,
        channelIds,
        channelNames
      });
      const result = await (0, _axios_utils.request)({
        axios: axiosInstance,
        method: 'post',
        url: `${_slack_api.SLACK_URL}chat.postMessage`,
        logger,
        data: {
          channel: channelToUse,
          text
        },
        headers,
        configurationUtilities,
        connectorUsageCollector
      });
      return buildSlackExecutorSuccessResponse({
        slackApiResponseData: result.data
      });
    } catch (error) {
      return buildSlackExecutorErrorResponse({
        slackApiError: error,
        logger
      });
    }
  };
  const postBlockkit = async ({
    channels = [],
    channelIds = [],
    channelNames = [],
    text
  }) => {
    try {
      const channelToUse = getChannelToUse({
        channels,
        channelIds,
        channelNames
      });
      const blockJson = JSON.parse(text);
      const result = await (0, _axios_utils.request)({
        axios: axiosInstance,
        method: 'post',
        url: `${_slack_api.SLACK_URL}chat.postMessage`,
        logger,
        data: {
          channel: channelToUse,
          blocks: blockJson.blocks
        },
        headers,
        configurationUtilities,
        connectorUsageCollector
      });
      return buildSlackExecutorSuccessResponse({
        slackApiResponseData: result.data
      });
    } catch (error) {
      return buildSlackExecutorErrorResponse({
        slackApiError: error,
        logger
      });
    }
  };
  return {
    validChannelId,
    postMessage,
    postBlockkit
  };
};
exports.createExternalService = createExternalService;