"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectorType = void 0;
var _common = require("@kbn/actions-plugin/common");
var _mustache_renderer = require("@kbn/actions-plugin/server/lib/mustache_renderer");
var _i18n = require("@kbn/i18n");
var _slack_api = require("@kbn/connector-schemas/slack_api");
var _api = require("./api");
var _service = require("./service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const supportedSubActions = ['getAllowedChannels', 'validChannelId', 'postMessage', 'postBlockkit'];
const getConnectorType = () => {
  return {
    id: _slack_api.CONNECTOR_ID,
    minimumLicenseRequired: 'gold',
    name: _slack_api.CONNECTOR_NAME,
    supportedFeatureIds: [_common.UptimeConnectorFeatureId, _common.AlertingConnectorFeatureId, _common.SecurityConnectorFeatureId, _common.WorkflowsConnectorFeatureId],
    validate: {
      config: {
        schema: _slack_api.SlackApiConfigSchema
      },
      secrets: {
        schema: _slack_api.SlackApiSecretsSchema,
        customValidator: validateSlackUrl
      },
      params: {
        schema: _slack_api.SlackApiParamsSchema
      }
    },
    renderParameterTemplates,
    executor: async execOptions => await slackApiExecutor(execOptions)
  };
};
exports.getConnectorType = getConnectorType;
const validateSlackUrl = (secretsObject, validatorServices) => {
  const {
    configurationUtilities
  } = validatorServices;
  try {
    configurationUtilities.ensureUriAllowed(_slack_api.SLACK_URL);
  } catch (allowedListError) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.slack_api.configurationError', {
      defaultMessage: 'error configuring slack action: {message}',
      values: {
        message: allowedListError.message
      }
    }));
  }
};
const renderParameterTemplates = (logger, params, variables) => {
  if (params.subAction === 'postMessage') {
    return {
      subAction: params.subAction,
      subActionParams: {
        ...params.subActionParams,
        text: (0, _mustache_renderer.renderMustacheString)(logger, params.subActionParams.text, variables, 'slack')
      }
    };
  } else if (params.subAction === 'postBlockkit') {
    return {
      subAction: params.subAction,
      subActionParams: {
        ...params.subActionParams,
        text: (0, _mustache_renderer.renderMustacheString)(logger, params.subActionParams.text, variables, 'json')
      }
    };
  }
  return params;
};
const slackApiExecutor = async ({
  actionId,
  config,
  params,
  secrets,
  configurationUtilities,
  logger,
  connectorUsageCollector
}) => {
  const subAction = params.subAction;
  if (!_api.api[subAction]) {
    const errorMessage = `[Action][ExternalService] -> [Slack API] Unsupported subAction type ${subAction}.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
  if (!supportedSubActions.includes(subAction)) {
    const errorMessage = `[Action][ExternalService] -> [Slack API] subAction ${subAction} not implemented.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
  const externalService = (0, _service.createExternalService)({
    config,
    secrets
  }, logger, configurationUtilities, connectorUsageCollector);
  if (subAction === 'validChannelId') {
    return await _api.api.validChannelId({
      externalService,
      params: params.subActionParams
    });
  }
  if (subAction === 'postMessage') {
    return await _api.api.postMessage({
      externalService,
      params: params.subActionParams
    });
  }
  if (subAction === 'postBlockkit') {
    return await _api.api.postBlockkit({
      externalService,
      params: params.subActionParams
    });
  }
  return {
    status: 'ok',
    data: {},
    actionId
  };
};