"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SentinelOneConnector = exports.API_PATH = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _sentinelone = require("@kbn/connector-schemas/sentinelone");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const API_PATH = exports.API_PATH = '/web/api/v2.1';
class SentinelOneConnector extends _server.SubActionConnector {
  constructor(params) {
    super(params);
    (0, _defineProperty2.default)(this, "urls", void 0);
    this.urls = {
      isolateHost: `${this.config.url}${API_PATH}/agents/actions/disconnect`,
      releaseHost: `${this.config.url}${API_PATH}/agents/actions/connect`,
      remoteScripts: `${this.config.url}${API_PATH}/remote-scripts`,
      remoteScriptStatus: `${this.config.url}${API_PATH}/remote-scripts/status`,
      remoteScriptsExecute: `${this.config.url}${API_PATH}/remote-scripts/execute`,
      remoteScriptsResults: `${this.config.url}${API_PATH}/remote-scripts/fetch-files`,
      agents: `${this.config.url}${API_PATH}/agents`,
      activities: `${this.config.url}${API_PATH}/activities`
    };
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.GET_REMOTE_SCRIPTS,
      method: 'getRemoteScripts',
      schema: _sentinelone.SentinelOneGetRemoteScriptsParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.FETCH_AGENT_FILES,
      method: 'fetchAgentFiles',
      schema: _sentinelone.SentinelOneFetchAgentFilesParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.DOWNLOAD_AGENT_FILE,
      method: 'downloadAgentFile',
      schema: _sentinelone.SentinelOneDownloadAgentFileParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.GET_ACTIVITIES,
      method: 'getActivities',
      schema: _sentinelone.SentinelOneGetActivitiesParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.GET_REMOTE_SCRIPT_STATUS,
      method: 'getRemoteScriptStatus',
      schema: _sentinelone.SentinelOneGetRemoteScriptStatusParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.GET_REMOTE_SCRIPT_RESULTS,
      method: 'getRemoteScriptResults',
      schema: _sentinelone.SentinelOneGetRemoteScriptResultsParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.DOWNLOAD_REMOTE_SCRIPT_RESULTS,
      method: 'downloadRemoteScriptResults',
      schema: _sentinelone.SentinelOneDownloadRemoteScriptResultsParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.GET_AGENTS,
      method: 'getAgents',
      schema: _sentinelone.SentinelOneGetAgentsParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.ISOLATE_HOST,
      method: 'isolateHost',
      schema: _sentinelone.SentinelOneIsolateHostParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.RELEASE_HOST,
      method: 'releaseHost',
      schema: _sentinelone.SentinelOneIsolateHostParamsSchema
    });
    this.registerSubAction({
      name: _sentinelone.SUB_ACTION.EXECUTE_SCRIPT,
      method: 'executeScript',
      schema: _sentinelone.SentinelOneExecuteScriptParamsSchema
    });
  }
  async fetchAgentFiles({
    files,
    agentId,
    zipPassCode
  }, connectorUsageCollector) {
    if (!agentId) {
      throw new Error(`'agentId' parameter is required`);
    }
    return this.sentinelOneApiRequest({
      url: `${this.urls.agents}/${agentId}/actions/fetch-files`,
      method: 'post',
      data: {
        data: {
          password: zipPassCode,
          files
        }
      },
      responseSchema: _sentinelone.SentinelOneFetchAgentFilesResponseSchema
    }, connectorUsageCollector);
  }
  async downloadAgentFile({
    agentId,
    activityId
  }, connectorUsageCollector) {
    if (!agentId) {
      throw new Error(`'agentId' parameter is required`);
    }
    return this.sentinelOneApiRequest({
      url: `${this.urls.agents}/${agentId}/uploads/${activityId}`,
      method: 'get',
      responseType: 'stream',
      responseSchema: _sentinelone.SentinelOneDownloadAgentFileResponseSchema
    }, connectorUsageCollector);
  }
  async getActivities(queryParams, connectorUsageCollector) {
    return this.sentinelOneApiRequest({
      url: this.urls.activities,
      method: 'get',
      params: queryParams,
      responseSchema: _sentinelone.SentinelOneGetActivitiesResponseSchema
    }, connectorUsageCollector);
  }
  async executeScript({
    filter,
    script
  }, connectorUsageCollector) {
    if (!filter.ids && !filter.uuids) {
      throw new Error(`A filter must be defined; either 'ids' or 'uuids'`);
    }
    return this.sentinelOneApiRequest({
      url: this.urls.remoteScriptsExecute,
      method: 'post',
      data: {
        data: {
          outputDestination: 'SentinelCloud',
          ...script
        },
        filter
      },
      responseSchema: _sentinelone.SentinelOneExecuteScriptResponseSchema
    }, connectorUsageCollector);
  }
  async isolateHost({
    alertIds,
    ...payload
  }, connectorUsageCollector) {
    const response = await this.getAgents(payload, connectorUsageCollector);
    if (response.data.length === 0) {
      const errorMessage = 'No agents found';
      throw new Error(errorMessage);
    }
    if (response.data[0].networkStatus === 'disconnected') {
      const errorMessage = 'Agent already isolated';
      throw new Error(errorMessage);
    }
    const agentId = response.data[0].id;
    return this.sentinelOneApiRequest({
      url: this.urls.isolateHost,
      method: 'post',
      data: {
        filter: {
          ids: agentId
        }
      },
      responseSchema: _sentinelone.SentinelOneIsolateHostResponseSchema
    }, connectorUsageCollector);
  }
  async releaseHost({
    alertIds,
    ...payload
  }, connectorUsageCollector) {
    const response = await this.getAgents(payload, connectorUsageCollector);
    if (response.data.length === 0) {
      throw new Error('No agents found');
    }
    if (response.data[0].networkStatus !== 'disconnected') {
      throw new Error('Agent not isolated');
    }
    const agentId = response.data[0].id;
    return this.sentinelOneApiRequest({
      url: this.urls.releaseHost,
      method: 'post',
      data: {
        filter: {
          ids: agentId
        }
      },
      responseSchema: _sentinelone.SentinelOneIsolateHostResponseSchema
    }, connectorUsageCollector);
  }
  async getAgents(payload, connectorUsageCollector) {
    return this.sentinelOneApiRequest({
      url: this.urls.agents,
      params: {
        ...payload
      },
      responseSchema: _sentinelone.SentinelOneGetAgentsResponseSchema
    }, connectorUsageCollector);
  }
  async getRemoteScriptStatus(payload, connectorUsageCollector) {
    return this.sentinelOneApiRequest({
      url: this.urls.remoteScriptStatus,
      params: {
        parent_task_id: payload.parentTaskId
      },
      responseSchema: _sentinelone.SentinelOneGetRemoteScriptStatusResponseSchema
    }, connectorUsageCollector);
  }
  async getRemoteScriptResults({
    taskIds
  }, connectorUsageCollector) {
    return this.sentinelOneApiRequest({
      url: this.urls.remoteScriptsResults,
      method: 'post',
      data: {
        data: {
          taskIds
        }
      },
      responseSchema: _sentinelone.SentinelOneGetRemoteScriptResultsResponseSchema
    }, connectorUsageCollector);
  }
  async downloadRemoteScriptResults({
    taskId
  }, connectorUsageCollector) {
    const scriptResultsInfo = await this.getRemoteScriptResults({
      taskIds: [taskId]
    }, connectorUsageCollector);
    this.logger.debug(() => `script results for taskId [${taskId}]:\n${JSON.stringify(scriptResultsInfo)}`);
    let fileUrl = '';
    for (const downloadLinkInfo of scriptResultsInfo.data.download_links) {
      if (downloadLinkInfo.taskId === taskId) {
        fileUrl = downloadLinkInfo.downloadUrl;
        break;
      }
    }
    if (!fileUrl) {
      throw new Error(`Download URL for script results of task id [${taskId}] not found`);
    }
    const downloadConnection = await this.request({
      url: fileUrl,
      method: 'get',
      responseType: 'stream',
      responseSchema: _sentinelone.SentinelOneDownloadRemoteScriptResultsResponseSchema
    }, connectorUsageCollector);
    return downloadConnection.data;
  }
  async sentinelOneApiRequest(req, connectorUsageCollector) {
    const response = await this.request({
      ...req,
      // We don't validate responses from SentinelOne API's because we do not want failures for cases
      // where the external system might add/remove/change values in the response that we have no
      // control over.
      responseSchema: _sentinelone.SentinelOneApiDoNotValidateResponsesSchema,
      params: {
        ...req.params,
        APIToken: this.secrets.token
      }
    }, connectorUsageCollector);
    return response.data;
  }
  getResponseErrorMessage(error) {
    var _error$response2, _error$response3;
    const appendResponseBody = message => {
      var _error$response$data, _error$response;
      const responseBody = JSON.stringify((_error$response$data = (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.data) !== null && _error$response$data !== void 0 ? _error$response$data : {});
      if (responseBody) {
        return `${message}\nResponse body: ${responseBody}`;
      }
      return message;
    };
    if (!((_error$response2 = error.response) !== null && _error$response2 !== void 0 && _error$response2.status)) {
      var _error$message;
      return appendResponseBody((_error$message = error.message) !== null && _error$message !== void 0 ? _error$message : 'Unknown API Error');
    }
    if (error.response.status === 401) {
      return appendResponseBody('Unauthorized API Error (401)');
    }
    return appendResponseBody(`API Error: [${(_error$response3 = error.response) === null || _error$response3 === void 0 ? void 0 : _error$response3.statusText}] ${error.message}`);
  }
  async getRemoteScripts(payload, connectorUsageCollector) {
    return this.sentinelOneApiRequest({
      url: this.urls.remoteScripts,
      params: {
        ...payload
      },
      responseSchema: _sentinelone.SentinelOneGetRemoteScriptsResponseSchema
    }, connectorUsageCollector);
  }
}
exports.SentinelOneConnector = SentinelOneConnector;