"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectorType = getConnectorType;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/actions-plugin/common");
var _task_running = require("@kbn/task-manager-plugin/server/task_running");
var _pagerduty = require("@kbn/connector-schemas/pagerduty");
var _utils = require("@kbn/connector-schemas/common/utils");
var _post_pagerduty = require("./post_pagerduty");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// connector type definition
function getConnectorType() {
  return {
    id: _pagerduty.CONNECTOR_ID,
    minimumLicenseRequired: 'gold',
    name: _pagerduty.CONNECTOR_NAME,
    supportedFeatureIds: [_common.AlertingConnectorFeatureId, _common.UptimeConnectorFeatureId, _common.SecurityConnectorFeatureId, _common.WorkflowsConnectorFeatureId],
    validate: {
      config: {
        schema: _pagerduty.ConfigSchema,
        customValidator: validateConnectorTypeConfig
      },
      secrets: {
        schema: _pagerduty.SecretsSchema
      },
      params: {
        schema: _pagerduty.ParamsSchema
      }
    },
    executor
  };
}
function validateConnectorTypeConfig(configObject, validatorServices) {
  const {
    configurationUtilities
  } = validatorServices;
  try {
    configurationUtilities.ensureUriAllowed(getPagerDutyApiUrl(configObject));
  } catch (allowListError) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.pagerduty.configurationError', {
      defaultMessage: 'error configuring pagerduty action: {message}',
      values: {
        message: allowListError.message
      }
    }));
  }
}
function getPagerDutyApiUrl(config) {
  return config.apiUrl || _pagerduty.PAGER_DUTY_API_URL;
}

// action executor

async function executor(execOptions) {
  const {
    actionId,
    config,
    secrets,
    params,
    services,
    configurationUtilities,
    logger,
    connectorUsageCollector
  } = execOptions;
  const apiUrl = getPagerDutyApiUrl(config);
  const headers = {
    'Content-Type': 'application/json',
    'X-Routing-Key': secrets.routingKey
  };
  const data = getBodyForEventAction(actionId, params);
  let response;
  try {
    response = await (0, _post_pagerduty.postPagerduty)({
      apiUrl,
      data,
      headers,
      services
    }, logger, configurationUtilities, connectorUsageCollector);
  } catch (err) {
    const message = _i18n.i18n.translate('xpack.stackConnectors.pagerduty.postingErrorMessage', {
      defaultMessage: 'error posting pagerduty event'
    });
    logger.warn(`error thrown posting pagerduty event: ${err.message}`);
    return {
      status: 'error',
      actionId,
      message,
      serviceMessage: err.message,
      errorSource: (0, _task_running.getErrorSource)(err)
    };
  }
  if (response == null) {
    const message = _i18n.i18n.translate('xpack.stackConnectors.pagerduty.unexpectedNullResponseErrorMessage', {
      defaultMessage: 'unexpected null response from pagerduty'
    });
    return {
      status: 'error',
      actionId,
      message
    };
  }
  logger.debug(`response posting pagerduty event: ${response.status}`);
  if (response.status === 202) {
    return {
      status: 'ok',
      actionId,
      data: response.data
    };
  }
  if (response.status === 429 || response.status >= 500) {
    const message = _i18n.i18n.translate('xpack.stackConnectors.pagerduty.postingRetryErrorMessage', {
      defaultMessage: 'error posting pagerduty event: http status {status}, retry later',
      values: {
        status: response.status
      }
    });
    return {
      status: 'error',
      actionId,
      message,
      retry: true
    };
  }
  const message = _i18n.i18n.translate('xpack.stackConnectors.pagerduty.postingUnexpectedErrorMessage', {
    defaultMessage: 'error posting pagerduty event: unexpected status {status}',
    values: {
      status: response.status
    }
  });
  return {
    status: 'error',
    actionId,
    message
  };
}

// utilities

const AcknowledgeOrResolve = new Set([_pagerduty.EVENT_ACTION_ACKNOWLEDGE, _pagerduty.EVENT_ACTION_RESOLVE]);
function getBodyForEventAction(actionId, params) {
  var _params$eventAction;
  const eventAction = (_params$eventAction = params.eventAction) !== null && _params$eventAction !== void 0 ? _params$eventAction : _pagerduty.EVENT_ACTION_TRIGGER;
  const data = {
    event_action: eventAction
  };
  if (params.dedupKey) {
    data.dedup_key = params.dedupKey;
  }

  // for acknowledge / resolve, just send the dedup key
  if (AcknowledgeOrResolve.has(eventAction)) {
    return data;
  }
  const convertedTimestamp = (0, _utils.convertTimestamp)(params.timestamp);
  data.payload = {
    summary: params.summary || 'No summary provided.',
    source: params.source || `Kibana Action ${actionId}`,
    severity: params.severity || 'info',
    ...(convertedTimestamp ? {
      timestamp: (0, _moment.default)(convertedTimestamp).toISOString()
    } : {}),
    ...(0, _lodash.omitBy)((0, _lodash.pick)(params, ['component', 'group', 'class']), _lodash.isUndefined),
    ...(params.customDetails ? {
      custom_details: params.customDetails
    } : {})
  };
  if (params.links) {
    data.links = params.links;
  }
  return data;
}