"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformApiUrlToRegex = exports.sanitizeRequest = exports.getRequestWithStreamOption = void 0;
var _openai = require("@kbn/connector-schemas/openai");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const APIS_ALLOWING_STREAMING = new Set([_openai.AZURE_OPENAI_CHAT_URL, _openai.AZURE_OPENAI_COMPLETIONS_URL, _openai.AZURE_OPENAI_COMPLETIONS_EXTENSIONS_URL]);

/**
 * Sanitizes the Azure Open AI request body to set stream to false
 * so users cannot specify a streaming response when the framework
 * is not prepared to handle streaming
 *
 * The stream parameter is only accepted in the Chat API, the Completion API
 * and the Completions Extensions API
 */
const sanitizeRequest = (url, body) => {
  return getRequestWithStreamOption(url, body, false);
};

/**
 * Intercepts the Azure Open AI request body to set the stream parameter
 *
 * The stream parameter is only accepted in the Chat API, the Completion API
 * and the Completions Extensions API
 */
exports.sanitizeRequest = sanitizeRequest;
const getRequestWithStreamOption = (url, body, stream) => {
  if (!Array.from(APIS_ALLOWING_STREAMING).map(apiUrl => transformApiUrlToRegex(apiUrl)).some(regex => url.match(regex) != null)) {
    return body;
  }
  try {
    const jsonBody = JSON.parse(body);
    if (jsonBody) {
      jsonBody.stream = stream;
      if (stream) {
        jsonBody.stream_options = {
          include_usage: true
        };
      }
    }
    return JSON.stringify(jsonBody);
  } catch (err) {
    // swallow the error
  }
  return body;
};
exports.getRequestWithStreamOption = getRequestWithStreamOption;
const transformApiUrlToRegex = apiUrl => {
  return new RegExp(apiUrl.replaceAll(`/`, `\\/`).replaceAll(`.`, `\\.`).replace(`{your-resource-name}`, `[^\\.\\/]+`).replace(`{deployment-id}`, `[^\\/]+`).replace(`?api-version={api-version}`, ``), 'g');
};
exports.transformApiUrlToRegex = transformApiUrlToRegex;