"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectorType = exports.configValidator = void 0;
var _i18n = require("@kbn/i18n");
var _types = require("@kbn/actions-plugin/server/sub_action_framework/types");
var _common = require("@kbn/actions-plugin/common");
var _server = require("@kbn/actions-plugin/server");
var _validators = require("@kbn/actions-plugin/server/sub_action_framework/helpers/validators");
var _openai = require("@kbn/connector-schemas/openai");
var _other_openai_utils = require("./lib/other_openai_utils");
var _openai2 = require("./openai");
var _render = require("./render");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getConnectorType = () => ({
  id: _openai.CONNECTOR_ID,
  name: _openai.CONNECTOR_NAME,
  getService: params => new _openai2.OpenAIConnector(params),
  schema: {
    config: _openai.ConfigSchema,
    secrets: _openai.SecretsSchema
  },
  validators: [{
    type: _types.ValidatorType.CONFIG,
    validator: configValidator
  }, {
    type: _types.ValidatorType.SECRETS,
    validator: secretsValidator
  }],
  supportedFeatureIds: [_common.GenerativeAIForSecurityConnectorFeatureId, _common.GenerativeAIForObservabilityConnectorFeatureId, _common.GenerativeAIForSearchPlaygroundConnectorFeatureId, _common.WorkflowsConnectorFeatureId],
  minimumLicenseRequired: 'enterprise',
  renderParameterTemplates: _render.renderParameterTemplates
});
exports.getConnectorType = getConnectorType;
const secretsValidator = secretsObject => {
  const validatorFn = 'certificateData' in secretsObject && secretsObject.certificateData || 'privateKeyData' in secretsObject && secretsObject.privateKeyData ? _other_openai_utils.pkiSecretsValidator : _other_openai_utils.nonPkiSecretsValidator;
  validatorFn(secretsObject);
  return secretsObject;
};
const configValidator = (configObject, validatorServices) => {
  try {
    (0, _validators.assertURL)(configObject.apiUrl);
    (0, _server.urlAllowListValidator)('apiUrl')(configObject, validatorServices);
    const {
      apiProvider
    } = configObject;
    if (apiProvider !== _openai.OpenAiProviderType.OpenAi && apiProvider !== _openai.OpenAiProviderType.AzureAi && apiProvider !== _openai.OpenAiProviderType.Other) {
      throw new Error(`API Provider is not supported${apiProvider && apiProvider.length ? `: ${apiProvider}` : ``}`);
    }
    return configObject;
  } catch (err) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.genAi.configurationErrorApiProvider', {
      defaultMessage: 'Error configuring OpenAI action: {err}',
      values: {
        err: err.toString()
      }
    }));
  }
};
exports.configValidator = configValidator;