"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectorType = getConnectorType;
var _common = require("@kbn/actions-plugin/common");
var _jira = require("@kbn/connector-schemas/jira");
var _validators = require("./validators");
var _service = require("./service");
var _api = require("./api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const supportedSubActions = ['getFields', 'getIncident', 'pushToService', 'issueTypes', 'fieldsByIssueType', 'issues', 'issue'];

// connector type definition
function getConnectorType() {
  return {
    id: _jira.CONNECTOR_ID,
    minimumLicenseRequired: 'gold',
    name: _jira.CONNECTOR_NAME,
    supportedFeatureIds: [_common.AlertingConnectorFeatureId, _common.CasesConnectorFeatureId, _common.UptimeConnectorFeatureId, _common.SecurityConnectorFeatureId, _common.WorkflowsConnectorFeatureId],
    validate: {
      config: {
        schema: _jira.ExternalIncidentServiceConfigurationSchema,
        customValidator: _validators.validate.config
      },
      secrets: {
        schema: _jira.ExternalIncidentServiceSecretConfigurationSchema,
        customValidator: _validators.validate.secrets
      },
      params: {
        schema: _jira.ExecutorParamsSchema
      }
    },
    executor
  };
}

// action executor
async function executor(execOptions) {
  var _data;
  const {
    actionId,
    config,
    params,
    secrets,
    configurationUtilities,
    logger,
    connectorUsageCollector
  } = execOptions;
  const {
    subAction,
    subActionParams
  } = params;
  let data = null;
  const externalService = (0, _service.createExternalService)({
    config,
    secrets
  }, logger, configurationUtilities, connectorUsageCollector);
  if (!_api.api[subAction]) {
    const errorMessage = `[Action][ExternalService] Unsupported subAction type ${subAction}.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
  if (!supportedSubActions.includes(subAction)) {
    const errorMessage = `[Action][ExternalService] subAction ${subAction} not implemented.`;
    logger.error(errorMessage);
    throw new Error(errorMessage);
  }
  if (subAction === 'getIncident') {
    const getIncidentParams = subActionParams;
    const res = await _api.api.getIncident({
      externalService,
      params: getIncidentParams
    });
    if (res != null) {
      data = res;
    }
  }
  if (subAction === 'pushToService') {
    const pushToServiceParams = subActionParams;
    data = await _api.api.pushToService({
      externalService,
      params: pushToServiceParams,
      logger
    });
    logger.debug(`response push to service for incident id: ${data.id}`);
  }
  if (subAction === 'issueTypes') {
    const getIssueTypesParams = subActionParams;
    data = await _api.api.issueTypes({
      externalService,
      params: getIssueTypesParams
    });
  }
  if (subAction === 'fieldsByIssueType') {
    const getFieldsByIssueTypeParams = subActionParams;
    data = await _api.api.fieldsByIssueType({
      externalService,
      params: getFieldsByIssueTypeParams
    });
  }
  if (subAction === 'getFields') {
    data = await _api.api.getFields({
      externalService,
      params: subActionParams
    });
  }
  if (subAction === 'issues') {
    const getIssuesParams = subActionParams;
    data = await _api.api.issues({
      externalService,
      params: getIssuesParams
    });
  }
  if (subAction === 'issue') {
    const getIssueParams = subActionParams;
    data = await _api.api.issue({
      externalService,
      params: getIssueParams
    });
  }
  return {
    status: 'ok',
    data: (_data = data) !== null && _data !== void 0 ? _data : {},
    actionId
  };
}