"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrowdstrikeConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _crowdstrike = require("@kbn/connector-schemas/crowdstrike");
var _rtr_session_manager = require("./rtr_session_manager");
var _token_manager = require("./token_manager");
var _types = require("./types");
var _error = require("./error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SUPPORTED_RTR_COMMANDS = ['runscript'];
const paramsSerializer = params => {
  return Object.entries(params).map(([key, value]) => `${encodeURIComponent(key)}=${encodeURIComponent(value)}`).join('&');
};

/**
 * Crowdstrike Connector
 * Uses instance-based token management with persistent storage via ConnectorTokenClient.
 * Each connector instance manages its own OAuth2 tokens with automatic refresh and caching.
 */

class CrowdstrikeConnector extends _server.SubActionConnector {
  constructor(params, experimentalFeatures) {
    super(params);
    (0, _defineProperty2.default)(this, "tokenManager", void 0);
    (0, _defineProperty2.default)(this, "experimentalFeatures", void 0);
    (0, _defineProperty2.default)(this, "crowdStrikeSessionManager", void 0);
    (0, _defineProperty2.default)(this, "urls", void 0);
    (0, _defineProperty2.default)(this, "crowdstrikeApiRequest", async (req, connectorUsageCollector, retried) => {
      this.logger.debug('Making CrowdStrike API request', {
        url: {
          full: req.url
        },
        method: req.method,
        retried: retried || false
      });
      try {
        this.logger.debug('Getting authentication token');
        const token = await this.tokenManager.get(connectorUsageCollector);
        const response = await this.request({
          ...req,
          // We don't validate responses from Crowdstrike API's because we do not want failures for cases
          // where the external system might add/remove/change values in the response that we have no
          // control over.
          responseSchema: _crowdstrike.CrowdstrikeApiDoNotValidateResponsesSchema,
          headers: {
            ...req.headers,
            Authorization: `Bearer ${token}`
          }
        }, connectorUsageCollector);
        this.logger.debug('CrowdStrike API request successful', {
          url: {
            full: req.url
          },
          status: response.status
        });
        return response.data;
      } catch (error) {
        var _error$response;
        this.logger.debug('CrowdStrike API request error', {
          url: {
            full: req.url
          },
          code: error.code,
          status: (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.status,
          message: error.message,
          retried: retried || false
        });
        if (error.code === 401 && !retried) {
          this.logger.debug('401 error, triggering token refresh and retry');
          await this.tokenManager.generateNew(connectorUsageCollector);
          return this.crowdstrikeApiRequest(req, connectorUsageCollector, true);
        }
        throw new _error.CrowdstrikeError(error.message);
      }
    });
    // Helper method to execute RTR commands with different API endpoints
    (0, _defineProperty2.default)(this, "executeRTRCommandWithUrl", async (url, payload, connectorUsageCollector) => {
      this.logger.debug('Executing RTR command', {
        url: {
          full: url
        },
        command: payload.command,
        endpointIds: payload.endpoint_ids
      });
      this.logger.debug('Initializing RTR session');
      const batchId = await this.crowdStrikeSessionManager.initializeSession({
        endpoint_ids: payload.endpoint_ids
      }, connectorUsageCollector);
      this.logger.debug(`RTR session initialized with batchId: ${batchId}`);
      const baseCommand = payload.command.split(' ')[0];
      if (!SUPPORTED_RTR_COMMANDS.includes(baseCommand)) {
        throw new _error.CrowdstrikeError('Command not supported');
      }
      return await this.crowdstrikeApiRequest({
        url,
        method: 'post',
        data: {
          base_command: baseCommand,
          command_string: payload.command,
          batch_id: batchId,
          hosts: payload.endpoint_ids,
          persist_all: false
        },
        paramsSerializer,
        responseSchema: _crowdstrike.CrowdstrikeExecuteRTRResponseSchema
      }, connectorUsageCollector);
    });
    this.experimentalFeatures = experimentalFeatures;
    this.urls = {
      getToken: `${this.config.url}/oauth2/token`,
      hostAction: `${this.config.url}/devices/entities/devices-actions/v2`,
      agents: `${this.config.url}/devices/entities/devices/v2`,
      agentStatus: `${this.config.url}/devices/entities/online-state/v1`,
      batchInitRTRSession: `${this.config.url}/real-time-response/combined/batch-init-session/v1`,
      batchRefreshRTRSession: `${this.config.url}/real-time-response/combined/batch-refresh-session/v1`,
      batchExecuteRTR: `${this.config.url}/real-time-response/combined/batch-command/v1`,
      batchActiveResponderExecuteRTR: `${this.config.url}/real-time-response/combined/batch-active-responder-command/v1`,
      batchAdminExecuteRTR: `${this.config.url}/real-time-response/combined/batch-admin-command/v1`,
      getRTRCloudScripts: `${this.config.url}/real-time-response/entities/scripts/v1`
    };

    // Initialize token manager
    this.tokenManager = new _token_manager.CrowdStrikeTokenManager({
      ...params,
      apiRequest: (req, connectorUsageCollector) => this.request(req, connectorUsageCollector)
    });
    this.crowdStrikeSessionManager = new _rtr_session_manager.CrowdStrikeSessionManager({
      batchInitRTRSession: this.urls.batchInitRTRSession,
      batchRefreshRTRSession: this.urls.batchRefreshRTRSession
    }, this.crowdstrikeApiRequest, this.logger);
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _crowdstrike.SUB_ACTION.GET_AGENT_DETAILS,
      method: 'getAgentDetails',
      schema: _crowdstrike.CrowdstrikeGetAgentsParamsSchema
    });
    this.registerSubAction({
      name: _crowdstrike.SUB_ACTION.HOST_ACTIONS,
      method: 'executeHostActions',
      schema: _crowdstrike.CrowdstrikeHostActionsParamsSchema
    });
    this.registerSubAction({
      name: _crowdstrike.SUB_ACTION.GET_AGENT_ONLINE_STATUS,
      method: 'getAgentOnlineStatus',
      schema: _crowdstrike.CrowdstrikeGetAgentsParamsSchema
    });
    if (this.experimentalFeatures.crowdstrikeConnectorRTROn) {
      this.registerSubAction({
        name: _crowdstrike.SUB_ACTION.EXECUTE_RTR_COMMAND,
        method: 'executeRTRCommand',
        schema: _crowdstrike.CrowdstrikeRTRCommandParamsSchema // Define a proper schema for the command
      });
      this.registerSubAction({
        name: _crowdstrike.SUB_ACTION.EXECUTE_ACTIVE_RESPONDER_RTR,
        method: 'batchActiveResponderExecuteRTR',
        schema: _crowdstrike.CrowdstrikeRTRCommandParamsSchema // Define a proper schema for the command
      });
      this.registerSubAction({
        name: _crowdstrike.SUB_ACTION.EXECUTE_ADMIN_RTR,
        method: 'batchAdminExecuteRTR',
        schema: _crowdstrike.CrowdstrikeRTRCommandParamsSchema // Define a proper schema for the command
      });
      this.registerSubAction({
        name: _crowdstrike.SUB_ACTION.GET_RTR_CLOUD_SCRIPTS,
        method: 'getRTRCloudScripts',
        schema: _crowdstrike.CrowdstrikeRTRCommandParamsSchema // Empty schema - this request do not have any parameters
      });
    }
  }
  async executeHostActions({
    alertIds,
    ...payload
  }, connectorUsageCollector) {
    return this.crowdstrikeApiRequest({
      url: this.urls.hostAction,
      method: 'post',
      params: {
        action_name: payload.command
      },
      data: {
        ids: payload.ids,
        ...(payload.actionParameters ? {
          action_parameters: Object.entries(payload.actionParameters).map(([name, value]) => ({
            name,
            value
          }))
        } : {})
      },
      paramsSerializer,
      responseSchema: _crowdstrike.CrowdstrikeHostActionsResponseSchema
    }, connectorUsageCollector);
  }
  async getAgentDetails(payload, connectorUsageCollector) {
    return this.crowdstrikeApiRequest({
      url: this.urls.agents,
      method: 'GET',
      params: {
        ids: payload.ids
      },
      paramsSerializer,
      responseSchema: _crowdstrike.RelaxedCrowdstrikeBaseApiResponseSchema
    }, connectorUsageCollector);
  }
  async getAgentOnlineStatus(payload, connectorUsageCollector) {
    return this.crowdstrikeApiRequest({
      url: this.urls.agentStatus,
      method: 'GET',
      params: {
        ids: payload.ids
      },
      paramsSerializer,
      responseSchema: _crowdstrike.RelaxedCrowdstrikeBaseApiResponseSchema
    }, connectorUsageCollector);
  }
  // Public method for generic RTR command execution
  async executeRTRCommand(payload, connectorUsageCollector) {
    return await this.executeRTRCommandWithUrl(this.urls.batchExecuteRTR, payload, connectorUsageCollector);
  }

  // Public method for Active Responder RTR command execution
  async batchActiveResponderExecuteRTR(payload, connectorUsageCollector) {
    return await this.executeRTRCommandWithUrl(this.urls.batchActiveResponderExecuteRTR, payload, connectorUsageCollector);
  }

  // Public method for Admin RTR command execution
  async batchAdminExecuteRTR(payload, connectorUsageCollector) {
    return await this.executeRTRCommandWithUrl(this.urls.batchAdminExecuteRTR, payload, connectorUsageCollector);
  }
  async getRTRCloudScripts(_payload, connectorUsageCollector) {
    return await this.crowdstrikeApiRequest({
      url: this.urls.getRTRCloudScripts,
      method: 'GET',
      paramsSerializer,
      responseSchema: _crowdstrike.CrowdstrikeGetScriptsResponseSchema
    }, connectorUsageCollector);
  }
  getResponseErrorMessage(error) {
    var _error$response2, _error$response2$data, _error$response2$data2, _error$response3, _error$response$data2;
    const errorData = (_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : (_error$response2$data = _error$response2.data) === null || _error$response2$data === void 0 ? void 0 : (_error$response2$data2 = _error$response2$data.errors) === null || _error$response2$data2 === void 0 ? void 0 : _error$response2$data2[0];
    if (errorData) {
      return errorData.message;
    }
    const cause = (0, _types.isAggregateError)(error.cause) ? error.cause.errors[0] : error.cause;
    if (cause) {
      // ENOTFOUND is the error code for when the host is unreachable eg. api.crowdstrike.com111
      if (cause.code === 'ENOTFOUND') {
        return `URL not found: ${cause.hostname}`;
      }
      // ECONNREFUSED is the error code for when the host is unreachable eg. http://MacBook-Pro-Tomasz.local:55555
      if (cause.code === 'ECONNREFUSED') {
        return `Connection Refused: ${cause.address}:${cause.port}`;
      }
    }
    if (!((_error$response3 = error.response) !== null && _error$response3 !== void 0 && _error$response3.status)) {
      var _error$response$data, _error$response4;
      return `Unknown API Error: ${JSON.stringify((_error$response$data = (_error$response4 = error.response) === null || _error$response4 === void 0 ? void 0 : _error$response4.data) !== null && _error$response$data !== void 0 ? _error$response$data : {})}`;
    }
    return `API Error: ${JSON.stringify((_error$response$data2 = error.response.data) !== null && _error$response$data2 !== void 0 ? _error$response$data2 : {})}`;
  }
}
exports.CrowdstrikeConnector = CrowdstrikeConnector;