"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateJson = exports.validateConnector = exports.validateCasesWebhookConfig = exports.validateAndNormalizeUrl = exports.normalizeURL = exports.ensureUriAllowed = exports.assertURL = void 0;
var _lodash = require("lodash");
var _auth = require("@kbn/connector-schemas/common/auth");
var i18n = _interopRequireWildcard(require("./translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validateCasesWebhookConfig = (configObject, validatorServices) => {
  const {
    configurationUtilities
  } = validatorServices;
  const {
    createCommentUrl,
    createIncidentUrl,
    viewIncidentUrl,
    getIncidentUrl,
    updateIncidentUrl
  } = configObject;
  const urls = [createIncidentUrl, createCommentUrl, viewIncidentUrl, getIncidentUrl, updateIncidentUrl];
  if (configObject.authType === _auth.AuthType.OAuth2ClientCredentials) {
    throw new Error(i18n.OAUTH2_NOT_SUPPORTED);
  }
  for (const url of urls) {
    if (url) {
      try {
        new URL(url);
      } catch (err) {
        throw new Error(i18n.INVALID_URL(err, url));
      }
      try {
        configurationUtilities.ensureUriAllowed(url);
      } catch (allowListError) {
        throw new Error(i18n.CONFIG_ERR(allowListError.message));
      }
    }
  }
};
exports.validateCasesWebhookConfig = validateCasesWebhookConfig;
const validateConnector = (configObject, secrets) => {
  if (configObject.hasAuth && (0, _lodash.isEmpty)(secrets)) return i18n.INVALID_AUTH;
  return null;
};
exports.validateConnector = validateConnector;
const validProtocols = ['http:', 'https:'];
const assertURL = url => {
  try {
    const parsedUrl = new URL(url);
    if (!parsedUrl.hostname) {
      throw new Error(`URL must contain hostname`);
    }
    if (!validProtocols.includes(parsedUrl.protocol)) {
      throw new Error(`Invalid protocol`);
    }
  } catch (error) {
    throw new Error(`${error.message}`);
  }
};
exports.assertURL = assertURL;
const ensureUriAllowed = (url, configurationUtilities) => {
  try {
    configurationUtilities.ensureUriAllowed(url);
  } catch (allowedListError) {
    throw Error(`${i18n.ALLOWED_HOSTS_ERROR(allowedListError.message)}`);
  }
};
exports.ensureUriAllowed = ensureUriAllowed;
const normalizeURL = url => {
  const urlWithoutTrailingSlash = url.endsWith('/') ? url.slice(0, -1) : url;
  const replaceDoubleSlashesRegex = new RegExp('([^:]/)/+', 'g');
  return urlWithoutTrailingSlash.replace(replaceDoubleSlashesRegex, '$1');
};
exports.normalizeURL = normalizeURL;
const validateAndNormalizeUrl = (url, configurationUtilities, urlDesc) => {
  try {
    assertURL(url);
    ensureUriAllowed(url, configurationUtilities);
    return normalizeURL(url);
  } catch (e) {
    throw Error(`Invalid ${urlDesc}: ${e}`);
  }
};
exports.validateAndNormalizeUrl = validateAndNormalizeUrl;
const validateJson = (jsonString, jsonDesc) => {
  try {
    JSON.parse(jsonString);
  } catch (e) {
    throw new Error(`JSON Error: ${jsonDesc} must be valid JSON`);
  }
};
exports.validateJson = validateJson;