"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ensureMessageFormat = void 0;
exports.extractRegionId = extractRegionId;
exports.formatBedrockBody = void 0;
exports.parseContent = parseContent;
exports.tee = tee;
exports.usesDeprecatedArguments = void 0;
var _eventstreamCodec = require("@smithy/eventstream-codec");
var _bedrock = require("@kbn/connector-schemas/bedrock");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatBedrockBody = ({
  messages,
  stopSequences,
  temperature = 0,
  system,
  maxTokens = _bedrock.DEFAULT_TOKEN_LIMIT,
  tools,
  toolChoice
}) => ({
  anthropic_version: 'bedrock-2023-05-31',
  ...ensureMessageFormat(messages, system),
  max_tokens: maxTokens,
  stop_sequences: stopSequences,
  temperature,
  tools,
  tool_choice: toolChoice
});
exports.formatBedrockBody = formatBedrockBody;
/**
 * Ensures that the messages are in the correct format for the Bedrock API
 * If 2 user or 2 assistant messages are sent in a row, Bedrock throws an error
 * We combine the messages into a single message to avoid this error
 * @param messages
 */
const ensureMessageFormat = (messages, systemPrompt) => {
  let system = systemPrompt ? systemPrompt : '';
  const newMessages = messages.reduce((acc, m) => {
    if (m.role === 'system') {
      system = `${system.length ? `${system}\n` : ''}${m.content}`;
      return acc;
    }
    const messageRole = () => ['assistant', 'ai'].includes(m.role) ? 'assistant' : 'user';
    if (m.rawContent) {
      acc.push({
        role: messageRole(),
        content: m.rawContent
      });
      return acc;
    }
    const lastMessage = acc[acc.length - 1];
    if (lastMessage && lastMessage.role === m.role && typeof lastMessage.content === 'string') {
      // Bedrock only accepts assistant and user roles.
      // If 2 user or 2 assistant messages are sent in a row, combine the messages into a single message
      return [...acc.slice(0, -1), {
        content: `${lastMessage.content}\n${m.content}`,
        role: m.role
      }];
    }

    // force role outside of system to ensure it is either assistant or user
    return [...acc, {
      content: m.content,
      role: messageRole()
    }];
  }, []);
  return system.length ? {
    system,
    messages: newMessages
  } : {
    messages: newMessages
  };
};
exports.ensureMessageFormat = ensureMessageFormat;
function parseContent(content) {
  let parsedContent = '';
  if (content.length === 1 && content[0].type === 'text' && content[0].text) {
    parsedContent = content[0].text;
  } else if (content.length > 1) {
    parsedContent = content.reduce((acc, {
      text
    }) => text ? `${acc}\n${text}` : acc, '');
  }
  return parsedContent;
}
const usesDeprecatedArguments = body => {
  var _JSON$parse;
  return ((_JSON$parse = JSON.parse(body)) === null || _JSON$parse === void 0 ? void 0 : _JSON$parse.prompt) != null;
};
exports.usesDeprecatedArguments = usesDeprecatedArguments;
function extractRegionId(url) {
  const match = (url !== null && url !== void 0 ? url : '').match(/https:\/\/.*?\.([a-z\-0-9]+)\.amazonaws\.com/);
  if (match) {
    return match[1];
  }
}

/**
 * Splits an async iterator into two independent async iterators which can be independently read from at different speeds.
 * @param asyncIterator The async iterator returned from Bedrock to split
 */
function tee(asyncIterator) {
  // @ts-ignore options is private, but we need it to create the new streams
  const streamOptions = asyncIterator.options;
  const streamLeft = new _eventstreamCodec.SmithyMessageDecoderStream(streamOptions);
  const streamRight = new _eventstreamCodec.SmithyMessageDecoderStream(streamOptions);

  // Queues to store chunks for each stream
  const leftQueue = [];
  const rightQueue = [];

  // Promises for managing when a chunk is available
  let leftPending = null;
  let rightPending = null;
  const distribute = async () => {
    for await (const chunk of asyncIterator) {
      // Push the chunk into both queues
      if (leftPending) {
        leftPending(chunk);
        leftPending = null;
      } else {
        leftQueue.push(chunk);
      }
      if (rightPending) {
        rightPending(chunk);
        rightPending = null;
      } else {
        rightQueue.push(chunk);
      }
    }

    // Signal the end of the iterator
    if (leftPending) {
      leftPending(null);
    }
    if (rightPending) {
      rightPending(null);
    }
  };

  // Start distributing chunks from the iterator
  distribute().catch(() => {
    // swallow errors
  });

  // Helper to create an async iterator for each stream
  const createIterator = (queue, setPending) => {
    return async function* () {
      while (true) {
        if (queue.length > 0) {
          yield queue.shift();
        } else {
          const chunk = await new Promise(resolve => setPending(resolve));
          if (chunk === null) break; // End of the stream
          yield chunk;
        }
      }
    };
  };

  // Assign independent async iterators to each stream
  streamLeft[Symbol.asyncIterator] = createIterator(leftQueue, fn => leftPending = fn);
  streamRight[Symbol.asyncIterator] = createIterator(rightQueue, fn => rightPending = fn);
  return [streamLeft, streamRight];
}