"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BedrockConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/actions-plugin/server");
var _api = require("@opentelemetry/api");
var _aws = _interopRequireDefault(require("aws4"));
var _clientBedrockRuntime = require("@aws-sdk/client-bedrock-runtime");
var _nodeHttpHandler = require("@smithy/node-http-handler");
var _stream = require("stream");
var _get_custom_agents = require("@kbn/actions-plugin/server/lib/get_custom_agents");
var _bedrock = require("@kbn/connector-schemas/bedrock");
var _create_gen_ai_dashboard = require("../lib/gen_ai/create_gen_ai_dashboard");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class BedrockConnector extends _server.SubActionConnector {
  constructor(params) {
    super(params);
    (0, _defineProperty2.default)(this, "url", void 0);
    (0, _defineProperty2.default)(this, "model", void 0);
    (0, _defineProperty2.default)(this, "bedrockClient", void 0);
    this.url = this.config.apiUrl;
    this.model = this.config.defaultModel;
    const {
      httpAgent,
      httpsAgent
    } = (0, _get_custom_agents.getCustomAgents)(this.configurationUtilities, this.logger, this.url);
    const isHttps = this.url.toLowerCase().startsWith('https');
    this.bedrockClient = new _clientBedrockRuntime.BedrockRuntimeClient({
      region: (0, _utils.extractRegionId)(this.config.apiUrl),
      credentials: {
        accessKeyId: this.secrets.accessKey,
        secretAccessKey: this.secrets.secret
      },
      requestHandler: new _nodeHttpHandler.NodeHttpHandler(isHttps ? {
        httpsAgent
      } : {
        httpAgent
      })
    });
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.RUN,
      method: 'runApi',
      schema: _bedrock.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.DASHBOARD,
      method: 'getDashboard',
      schema: _bedrock.DashboardActionParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.TEST,
      method: 'runApi',
      schema: _bedrock.RunActionParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.INVOKE_AI,
      method: 'invokeAI',
      schema: _bedrock.InvokeAIActionParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.INVOKE_STREAM,
      method: 'invokeStream',
      schema: _bedrock.InvokeAIActionParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.INVOKE_AI_RAW,
      method: 'invokeAIRaw',
      schema: _bedrock.InvokeAIRawActionParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.BEDROCK_CLIENT_SEND,
      method: 'bedrockClientSend',
      schema: _bedrock.BedrockClientSendParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.CONVERSE,
      method: 'converse',
      schema: _bedrock.ConverseActionParamsSchema
    });
    this.registerSubAction({
      name: _bedrock.SUB_ACTION.CONVERSE_STREAM,
      method: 'converseStream',
      schema: _bedrock.ConverseStreamActionParamsSchema
    });
  }
  getResponseErrorMessage(error) {
    var _error$response, _error$response2, _error$response2$data, _error$response4, _error$response5, _error$response5$data;
    if (!((_error$response = error.response) !== null && _error$response !== void 0 && _error$response.status)) {
      var _error$code, _error$message;
      return `Unexpected API Error: ${(_error$code = error.code) !== null && _error$code !== void 0 ? _error$code : ''} - ${(_error$message = error.message) !== null && _error$message !== void 0 ? _error$message : 'Unknown error'}`;
    }
    if (error.response.status === 400 && ((_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : (_error$response2$data = _error$response2.data) === null || _error$response2$data === void 0 ? void 0 : _error$response2$data.message) === 'The requested operation is not recognized by the service.') {
      // Leave space in the string below, \n is not being rendered in the UI
      return `API Error: ${error.response.data.message}

The Kibana Connector in use may need to be reconfigured with an updated Amazon Bedrock endpoint, like \`bedrock-runtime\`.`;
    }
    if (error.response.status === 401) {
      var _error$response3, _error$response3$data;
      return `Unauthorized API Error${(_error$response3 = error.response) !== null && _error$response3 !== void 0 && (_error$response3$data = _error$response3.data) !== null && _error$response3$data !== void 0 && _error$response3$data.message ? `: ${error.response.data.message}` : ''}`;
    }
    return `API Error: ${(_error$response4 = error.response) === null || _error$response4 === void 0 ? void 0 : _error$response4.statusText}${(_error$response5 = error.response) !== null && _error$response5 !== void 0 && (_error$response5$data = _error$response5.data) !== null && _error$response5$data !== void 0 && _error$response5$data.message ? ` - ${error.response.data.message}` : ''}`;
  }

  /**
   * provides the AWS signature to the external API endpoint
   * @param body The request body to be signed.
   * @param path The path of the request URL.
   */
  signRequest(body, path, stream) {
    const {
      host
    } = new URL(this.url);
    return _aws.default.sign({
      host,
      headers: stream ? {
        accept: 'application/vnd.amazon.eventstream',
        'Content-Type': 'application/json',
        'x-amzn-bedrock-accept': '*/*'
      } : {
        'Content-Type': 'application/json',
        Accept: '*/*'
      },
      body,
      path,
      // Despite AWS docs, this value does not always get inferred. We need to always send it
      service: 'bedrock'
    }, {
      secretAccessKey: this.secrets.secret,
      accessKeyId: this.secrets.accessKey
    });
  }

  /**
   *  retrieves a dashboard from the Kibana server and checks if the
   *  user has the necessary privileges to access it.
   * @param dashboardId The ID of the dashboard to retrieve.
   */
  async getDashboard({
    dashboardId
  }) {
    const privilege = await this.esClient.transport.request({
      path: '/_security/user/_has_privileges',
      method: 'POST',
      body: {
        index: [{
          names: ['.kibana-event-log-*'],
          allow_restricted_indices: true,
          privileges: ['read']
        }]
      }
    });
    if (!(privilege !== null && privilege !== void 0 && privilege.has_all_requested)) {
      return {
        available: false
      };
    }
    const response = await (0, _create_gen_ai_dashboard.initDashboard)({
      logger: this.logger,
      savedObjectsClient: this.savedObjectsClient,
      dashboardId,
      genAIProvider: 'Bedrock'
    });
    return {
      available: response.success
    };
  }
  async runApiRaw(params, connectorUsageCollector) {
    const response = await this.request(params, connectorUsageCollector);
    return response.data;
  }
  async runApiLatest(params, connectorUsageCollector) {
    const response = await this.request(params, connectorUsageCollector);
    // keeping the response the same as claude 2 for our APIs
    // adding the usage object for better token tracking
    return {
      completion: (0, _utils.parseContent)(response.data.content),
      stop_reason: response.data.stop_reason,
      usage: response.data.usage
    };
  }

  /**
   * responsible for making a POST request to the external API endpoint and returning the response data
   * @param body The stringified request body to be sent in the POST request.
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   * @param signal Optional signal to cancel the request.
   * @param timeout Optional timeout for the request.
   * @param raw Optional flag to indicate if the response should be returned as raw data.
   */
  async runApi({
    body,
    model: reqModel,
    signal,
    timeout,
    raw
  }, connectorUsageCollector) {
    const parentSpan = _api.trace.getActiveSpan();
    parentSpan === null || parentSpan === void 0 ? void 0 : parentSpan.setAttribute('bedrock.raw_request', body);
    // set model on per request basis
    // Application Inference Profile IDs need to be encoded when using the API
    // Decode first to ensure an existing encoded value is not double encoded
    const model = reqModel !== null && reqModel !== void 0 ? reqModel : this.model;
    if (!model) {
      throw new Error('No model specified. Please configure a default model.');
    }
    const currentModel = encodeURIComponent(decodeURIComponent(model));
    const path = `/model/${currentModel}/invoke`;
    const signed = this.signRequest(body, path, false);
    const requestArgs = {
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      data: body,
      signal,
      // give up to 2 minutes for response
      timeout: timeout !== null && timeout !== void 0 ? timeout : _bedrock.DEFAULT_TIMEOUT_MS
    };
    if (raw) {
      return this.runApiRaw({
        ...requestArgs,
        responseSchema: _bedrock.InvokeAIRawActionResponseSchema
      }, connectorUsageCollector);
    }
    // possible api received deprecated arguments, which will still work with the deprecated Claude 2 models
    if ((0, _utils.usesDeprecatedArguments)(body)) {
      return this.runApiRaw({
        ...requestArgs,
        responseSchema: _bedrock.RunActionResponseSchema
      }, connectorUsageCollector);
    }
    return this.runApiLatest({
      ...requestArgs,
      responseSchema: _bedrock.RunApiLatestResponseSchema
    }, connectorUsageCollector);
  }

  /**
   *  NOT INTENDED TO BE CALLED DIRECTLY
   *  call invokeStream instead
   *  responsible for making a POST request to a specified URL with a given request body.
   *  The response is then processed based on whether it is a streaming response or a regular response.
   * @param body The stringified request body to be sent in the POST request.
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async streamApi({
    body,
    model: reqModel,
    signal,
    timeout
  }, connectorUsageCollector) {
    const parentSpan = _api.trace.getActiveSpan();
    parentSpan === null || parentSpan === void 0 ? void 0 : parentSpan.setAttribute('bedrock.raw_request', body);
    // set model on per request basis
    // Application Inference Profile IDs need to be encoded when using the API
    // Decode first to ensure an existing encoded value is not double encoded
    const model = reqModel !== null && reqModel !== void 0 ? reqModel : this.model;
    if (!model) {
      throw new Error('No model specified. Please configure a default model.');
    }
    const currentModel = encodeURIComponent(decodeURIComponent(model));
    const path = `/model/${currentModel}/invoke-with-response-stream`;
    const signed = this.signRequest(body, path, true);
    const response = await this.request({
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      responseSchema: _bedrock.StreamingResponseSchema,
      data: body,
      responseType: 'stream',
      signal,
      timeout
    }, connectorUsageCollector);
    return response.data.pipe(new _stream.PassThrough());
  }

  /**
   *  takes in an array of messages and a model as inputs. It calls the streamApi method to make a
   *  request to the Bedrock API with the formatted messages and model. It then returns a Transform stream
   *  that pipes the response from the API through the transformToString function,
   *  which parses the proprietary response into a string of the response text alone
   * @param messages An array of messages to be sent to the API
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   */
  async invokeStream({
    messages,
    model,
    stopSequences,
    system,
    temperature,
    signal,
    timeout,
    tools,
    toolChoice
  }, connectorUsageCollector) {
    const res = await this.streamApi({
      body: JSON.stringify((0, _utils.formatBedrockBody)({
        messages,
        stopSequences,
        system,
        temperature,
        tools,
        toolChoice
      })),
      model,
      signal,
      timeout
    }, connectorUsageCollector);
    return res;
  }

  /**
   * Non-streamed security solution AI Assistant requests
   * Responsible for invoking the runApi method with the provided body.
   * It then formats the response into a string
   * @param messages An array of messages to be sent to the API
   * @param model Optional model to be used for the API request. If not provided, the default model from the connector will be used.
   * @returns an object with the response string as a property called message
   */
  async invokeAI({
    messages,
    model,
    stopSequences,
    system,
    temperature,
    maxTokens,
    signal,
    timeout,
    tools,
    toolChoice
  }, connectorUsageCollector) {
    const res = await this.runApi({
      body: JSON.stringify((0, _utils.formatBedrockBody)({
        messages,
        stopSequences,
        system,
        temperature,
        maxTokens,
        tools,
        toolChoice
      })),
      model,
      signal,
      timeout
    }, connectorUsageCollector);
    return {
      message: res.completion.trim(),
      usage: res === null || res === void 0 ? void 0 : res.usage
    };
  }
  async invokeAIRaw({
    messages,
    model,
    stopSequences,
    system,
    temperature,
    maxTokens = _bedrock.DEFAULT_TOKEN_LIMIT,
    signal,
    timeout,
    tools,
    toolChoice,
    anthropicVersion
  }, connectorUsageCollector) {
    const res = await this.runApi({
      body: JSON.stringify({
        messages,
        stop_sequences: stopSequences,
        system,
        temperature,
        max_tokens: maxTokens,
        tools,
        tool_choice: toolChoice,
        anthropic_version: anthropicVersion
      }),
      model,
      signal,
      timeout,
      raw: true
    }, connectorUsageCollector);
    return res;
  }

  /**
   * Sends a request via the BedrockRuntimeClient to perform a conversation action.
   * @param params - The parameters for the conversation action.
   * @param params.signal - The signal to cancel the request.
   * @param params.command - The command class to be sent to the API. (ConverseCommand | ConverseStreamCommand)
   * @param connectorUsageCollector - The usage collector for the connector.
   * @returns A promise that resolves to the response of the conversation action.
   */
  async bedrockClientSend({
    signal,
    command
  }, connectorUsageCollector) {
    if (command.input.modelId === 'preconfigured') {
      command.input.modelId = this.model;
    }
    connectorUsageCollector.addRequestBodyBytes(undefined, command);
    const res = await this.bedrockClient.send(command, {
      abortSignal: signal
    });
    if ('stream' in res) {
      const resultStream = res.stream;
      // splits the stream in two, [stream = consumer, tokenStream = token tracking]
      const [stream, tokenStream] = (0, _utils.tee)(resultStream);
      return {
        ...res,
        stream,
        tokenStream
      };
    }
    return res;
  }

  /**
   * Implements support for Bedrock's converse API which provides a simpler interface for single-turn conversations
   * Adapted from invokeAI and runApi to use the native Bedrock converse API endpoint
   * @param params Conversation parameters including messages, model, etc.
   * @param connectorUsageCollector Collector for usage metrics
   * @returns A promise that resolves to the conversation response
   */
  async converse({
    messages,
    model: reqModel,
    stopSequences,
    system,
    temperature,
    maxTokens,
    tools,
    toolChoice,
    signal,
    timeout = _bedrock.DEFAULT_TIMEOUT_MS
  }, connectorUsageCollector) {
    const modelId = reqModel !== null && reqModel !== void 0 ? reqModel : this.model;
    if (!modelId) {
      throw new Error('No model specified. Please configure a default model.');
    }
    const currentModel = encodeURIComponent(decodeURIComponent(modelId));
    const path = `/model/${currentModel}/converse`;
    const request = {
      messages,
      inferenceConfig: {
        temperature,
        stopSequences,
        maxTokens
      },
      toolConfig: {
        tools,
        toolChoice: {
          auto: toolChoice
        }
      },
      system,
      modelId
    };
    const requestBody = JSON.stringify(request);
    const signed = this.signRequest(requestBody, path, true);
    const requestArgs = {
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      data: requestBody,
      signal,
      timeout,
      responseSchema: _bedrock.RunApiLatestResponseSchema
    };
    const response = await this.runApiLatest(requestArgs, connectorUsageCollector);
    return response;
  }
  async _converseStream({
    messages,
    model: reqModel,
    stopSequences,
    system,
    temperature,
    maxTokens,
    tools,
    toolChoice,
    signal,
    timeout = _bedrock.DEFAULT_TIMEOUT_MS,
    connectorUsageCollector
  }) {
    const modelId = reqModel !== null && reqModel !== void 0 ? reqModel : this.model;
    if (!modelId) {
      throw new Error('No model specified. Please configure a default model.');
    }
    const currentModel = encodeURIComponent(decodeURIComponent(modelId));
    const path = `/model/${currentModel}/converse-stream`;
    const request = {
      messages,
      inferenceConfig: {
        temperature,
        stopSequences,
        maxTokens
      },
      toolConfig: {
        tools,
        toolChoice
      },
      system,
      modelId
    };
    const requestBody = JSON.stringify(request);
    const signed = this.signRequest(requestBody, path, true);
    const parentSpan = _api.trace.getActiveSpan();
    parentSpan === null || parentSpan === void 0 ? void 0 : parentSpan.setAttribute('bedrock.raw_request', requestBody);
    const response = await this.request({
      ...signed,
      url: `${this.url}${path}`,
      method: 'post',
      responseSchema: _bedrock.StreamingResponseSchema,
      data: requestBody,
      responseType: 'stream',
      signal,
      timeout
    }, connectorUsageCollector);
    if (response.data) {
      const resultStream = response.data;
      // splits the stream in two, [stream = consumer, tokenStream = token tracking]
      const [stream, tokenStream] = (0, _utils.tee)(resultStream);
      return {
        ...response.data,
        stream: _stream.Readable.from(stream).pipe(new _stream.PassThrough()),
        tokenStream: _stream.Readable.from(tokenStream).pipe(new _stream.PassThrough())
      };
    }
    return response;
  }

  /**
   * Implements support for Bedrock's converse-stream API which provides a streaming interface for conversations
   * Adapted from invokeStream and streamApi to use the native Bedrock converse-stream API endpoint
   * @param params Conversation parameters including messages, model, etc.
   * @param connectorUsageCollector Collector for usage metrics
   * @returns A streaming response as an IncomingMessage
   */
  async converseStream({
    messages,
    model: reqModel,
    stopSequences,
    system,
    temperature,
    maxTokens,
    tools,
    toolChoice,
    signal,
    timeout = _bedrock.DEFAULT_TIMEOUT_MS
  }, connectorUsageCollector) {
    return await this._converseStream({
      messages,
      model: reqModel,
      stopSequences,
      system,
      temperature,
      maxTokens,
      tools,
      toolChoice,
      signal,
      timeout,
      connectorUsageCollector
    });
  }
}
exports.BedrockConnector = BedrockConnector;