"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formSerializer = exports.formDeserializer = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * The serializer and deserializer are needed to transform the headers of
 * the webhook connectors. The webhook connector uses the UseArray component
 * to add dynamic headers to the form. The UseArray component formats the fields
 * as an array of objects. The schema for the headers of the webhook connector
 * is Record<string, string>. We need to transform the UseArray format to the one
 * accepted by the backend. At the moment, the UseArray does not accept
 * a serializer and deserializer so it has to be done on the form level.
 */

const formDeserializer = data => {
  var _data$config$headers, _data$config;
  if (!data.actionTypeId) {
    // Hook form lib can call deserializer *also* while editing the form (indicated by actionTypeId
    // still being undefined). Changing the reference of form data subproperties causes problems
    // with the UseArray that is used to edit the headers. For this reason, we leave the data unchanged.
    return data;
  }
  const configHeaders = Object.entries((_data$config$headers = data === null || data === void 0 ? void 0 : (_data$config = data.config) === null || _data$config === void 0 ? void 0 : _data$config.headers) !== null && _data$config$headers !== void 0 ? _data$config$headers : {}).map(([key, value]) => ({
    key,
    value,
    type: 'config'
  }));
  return {
    ...data,
    config: {
      ...data.config,
      headers: (0, _lodash.isEmpty)(configHeaders) ? undefined : configHeaders
    },
    __internal__: {
      headers: configHeaders
    }
  };
};
exports.formDeserializer = formDeserializer;
const buildHeaderRecords = (headers, type) => {
  return headers.filter(header => header.type === type && header.key && header.key.trim()).reduce((acc, {
    key,
    value
  }) => {
    acc[key] = value;
    return acc;
  }, {});
};
const formSerializer = formData => {
  var _formData$__internal_, _formData$__internal_2;
  const headers = (_formData$__internal_ = formData === null || formData === void 0 ? void 0 : (_formData$__internal_2 = formData.__internal__) === null || _formData$__internal_2 === void 0 ? void 0 : _formData$__internal_2.headers) !== null && _formData$__internal_ !== void 0 ? _formData$__internal_ : [];
  const configHeaders = buildHeaderRecords(headers, 'config');
  const secretHeaders = buildHeaderRecords(headers, 'secret');
  return {
    ...formData,
    config: {
      ...formData.config,
      headers: (0, _lodash.isEmpty)(configHeaders) ? null : configHeaders
    },
    secrets: {
      ...formData.secrets,
      secretHeaders: (0, _lodash.isEmpty)(secretHeaders) ? undefined : secretHeaders
    }
  };
};
exports.formSerializer = formSerializer;