"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.supertestToObservable = supertestToObservable;
var _stream = require("stream");
var _eventsourceParser = require("eventsource-parser");
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Convert a supertest response to an SSE observable.
 *
 * Note: the supertest response should *NOT* be awaited when using that utility,
 * or at least not before calling it.
 *
 * @example
 * ```ts
 * const response = supertest
 *   .post(`/some/sse/endpoint`)
 *   .set('kbn-xsrf', 'kibana')
 *   .send({
 *     some: 'thing'
 *   });
 * const events = supertestIntoObservable(response);
 * ```
 */
function supertestToObservable(response) {
  const stream = new _stream.PassThrough();
  response.pipe(stream);
  return new _rxjs.Observable(subscriber => {
    const parser = (0, _eventsourceParser.createParser)({
      onEvent: event => {
        var _event$event;
        subscriber.next({
          type: (_event$event = event.event) !== null && _event$event !== void 0 ? _event$event : 'event',
          ...JSON.parse(event.data)
        });
      }
    });
    const readStream = async () => {
      return new Promise((resolve, reject) => {
        const decoder = new TextDecoder();
        const processChunk = value => {
          parser.feed(decoder.decode(value, {
            stream: true
          }));
        };
        stream.on('data', chunk => {
          processChunk(chunk);
        });
        stream.on('end', () => resolve());
        stream.on('error', err => reject(err));
      });
    };
    readStream().then(() => {
      subscriber.complete();
    }).catch(error => {
      subscriber.error(error);
    });
  });
}