"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GetSLOHealth = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LAG_THRESHOLD_MINUTES = 10;
const STALE_THRESHOLD_MINUTES = 2 * 24 * 60;
class GetSLOHealth {
  constructor(scopedClusterClient, repository) {
    this.scopedClusterClient = scopedClusterClient;
    this.repository = repository;
  }
  async execute(params) {
    const sloIds = (0, _lodash.uniq)((0, _lodash.map)(params.list, 'sloId'));
    const definitions = await this.repository.findAllByIds(sloIds);
    const definitionById = (0, _lodash.keyBy)(definitions, 'id');
    const list = params.list.filter(item => !!definitionById[item.sloId]).map(item => ({
      id: item.sloId,
      instanceId: item.sloInstanceId,
      revision: definitionById[item.sloId].revision,
      name: definitionById[item.sloId].name
    }));
    const [summaryDocsById, transformStatsById] = await Promise.all([this.getSummaryDocsById(list), this.getTransformStats(definitions)]);
    const results = list.map(item => {
      const health = computeHealth(transformStatsById, item);
      const state = computeState(summaryDocsById, item);
      return {
        sloId: item.id,
        sloName: item.name,
        sloInstanceId: item.instanceId,
        sloRevision: item.revision,
        state,
        health
      };
    });
    return _sloSchema.fetchSLOHealthResponseSchema.encode(results);
  }
  async getSummaryDocsById(list) {
    const summaryDocs = await this.scopedClusterClient.asCurrentUser.search({
      index: _constants.SUMMARY_DESTINATION_INDEX_PATTERN,
      query: {
        bool: {
          should: list.map(item => ({
            bool: {
              must: [{
                term: {
                  'slo.id': item.id
                }
              }, {
                term: {
                  'slo.instanceId': item.instanceId
                }
              }, {
                term: {
                  'slo.revision': item.revision
                }
              }]
            }
          }))
        }
      }
    });
    const summaryDocsById = (0, _lodash.groupBy)(summaryDocs.hits.hits.map(hit => hit._source), doc => buildSummaryKey(doc.slo.id, doc.slo.instanceId));
    return summaryDocsById;
  }
  async getTransformStats(definitions) {
    const stats = await this.scopedClusterClient.asSecondaryAuthUser.transform.getTransformStats({
      transform_id: definitions.map(definition => (0, _constants.getWildcardTransformId)(definition.id, definition.revision)),
      allow_no_match: true,
      size: definitions.length * 2
    }, {
      ignore: [404]
    });
    return (0, _lodash.keyBy)(stats.transforms, transform => transform.id);
  }
}
exports.GetSLOHealth = GetSLOHealth;
function buildSummaryKey(id, instanceId) {
  return id + '|' + instanceId;
}
function computeState(summaryDocsById, item) {
  const sloSummaryDocs = summaryDocsById[buildSummaryKey(item.id, item.instanceId)];
  let state = 'no_data';
  if (!sloSummaryDocs) {
    return state;
  }
  const hasOnlyTempSummaryDoc = sloSummaryDocs.every(doc => doc.isTempDoc); // only temporary documents mean the summary transform did not run yet
  const sloSummarydoc = sloSummaryDocs.find(doc => !doc.isTempDoc);
  const latestSliTimestamp = sloSummarydoc === null || sloSummarydoc === void 0 ? void 0 : sloSummarydoc.latestSliTimestamp;
  const summaryUpdatedAt = sloSummarydoc === null || sloSummarydoc === void 0 ? void 0 : sloSummarydoc.summaryUpdatedAt;
  if (hasOnlyTempSummaryDoc) {
    state = 'no_data';
  } else if (summaryUpdatedAt && latestSliTimestamp) {
    const summaryLag = (0, _moment.default)().diff(new Date(summaryUpdatedAt), 'minute');
    const indexingLag = (0, _moment.default)(summaryUpdatedAt).diff(new Date(latestSliTimestamp), 'minute');

    // When the summaryUpdatedAt is greater than STALE_THRESHOLD_MINUTES minutes, the SLO is considered stale since no new data triggered a summary document update.
    // When the difference between the summaryUpdatedAt and the latestSliTimestamp is
    // - Below LAG_THRESHOLD_MINUTES minutes, the SLO has cought up with the sli data, and is running correctly
    // - Above LAG_THRESHOLD_MINUTES minutes, the SLO is indexing
    if (summaryLag > STALE_THRESHOLD_MINUTES) {
      state = 'stale';
    } else {
      state = indexingLag >= LAG_THRESHOLD_MINUTES ? 'indexing' : 'running';
    }
  }
  return state;
}
function getTransformHealth(transformStat) {
  var _transformStat$state, _transformStat$health, _transformStat$health2;
  if (!transformStat) {
    return {
      status: 'missing'
    };
  }
  const transformState = (_transformStat$state = transformStat.state) === null || _transformStat$state === void 0 ? void 0 : _transformStat$state.toLowerCase();
  return ((_transformStat$health = transformStat.health) === null || _transformStat$health === void 0 ? void 0 : (_transformStat$health2 = _transformStat$health.status) === null || _transformStat$health2 === void 0 ? void 0 : _transformStat$health2.toLowerCase()) === 'green' ? {
    status: 'healthy',
    transformState: transformState
  } : {
    status: 'unhealthy',
    transformState: transformState
  };
}
function computeHealth(transformStatsById, item) {
  const rollup = getTransformHealth(transformStatsById[(0, _constants.getSLOTransformId)(item.id, item.revision)]);
  const summary = getTransformHealth(transformStatsById[(0, _constants.getSLOSummaryTransformId)(item.id, item.revision)]);
  const overall = rollup.status === 'healthy' && rollup.transformState === 'started' && summary.status === 'healthy' && summary.transformState === 'started' ? 'healthy' : 'unhealthy';
  return {
    overall,
    rollup,
    summary
  };
}