"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.requestEndpointFieldsSearch = exports.endpointFieldsProvider = void 0;
var _rxjs = require("rxjs");
var _index_fields = require("@kbn/timelines-plugin/server/search_strategy/index_fields");
var _constants = require("../../../common/endpoint/constants");
var _errors = require("../../endpoint/errors");
var _parse_request = require("./parse_request");
var _index_name_utilities = require("../../../common/endpoint/utils/index_name_utilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * EndpointFieldProvider mimics indexField provider from timeline plugin: x-pack/platform/plugins/shared/timelines/server/search_strategy/index_fields/index.ts
 * but it uses ES internalUser instead to avoid adding extra index privileges for users with event filters permissions.
 * It is used to retrieve index patterns for event filters form.
 */
const endpointFieldsProvider = (context, indexPatterns) => {
  // require the fields once we actually need them, rather than ahead of time, and pass
  // them to createFieldItem to reduce the amount of work done as much as possible
  const beatFields =
  // eslint-disable-next-line @typescript-eslint/no-var-requires
  require('@kbn/timelines-plugin/server/utils/beat_schema/fields.json').fieldsBeat;
  return {
    search: (request, _, deps) => (0, _rxjs.from)(requestEndpointFieldsSearch(context, request, deps, beatFields, indexPatterns))
  };
};
exports.endpointFieldsProvider = endpointFieldsProvider;
const requestEndpointFieldsSearch = async (context, request, deps, beatFields, indexPatterns) => {
  const isTAAdvancedModeFeatureFlagEnabled = context.experimentalFeatures.trustedAppsAdvancedMode;
  let parsedRequest = (0, _parse_request.parseRequest)(request);
  if (parsedRequest.indices.length > 1 || parsedRequest.indices[0] !== _constants.eventsIndexPattern && parsedRequest.indices[0] !== _constants.METADATA_UNITED_INDEX) {
    throw new Error(`Invalid indices request ${request.indices.join(', ')}`);
  }
  if (parsedRequest.indices[0] === _constants.eventsIndexPattern) {
    const {
      id: spaceId
    } = await context.getActiveSpace(deps.request);
    const integrationNamespaces = await context.getInternalFleetServices(spaceId).getIntegrationNamespaces(['endpoint']);
    const namespaces = integrationNamespaces.endpoint;
    if (namespaces && namespaces.length > 0) {
      const combinedPatterns = namespaces.map(namespace => (0, _index_name_utilities.buildIndexNameWithNamespace)(_constants.eventsIndexPattern, namespace, {
        preserveWildcard: true
      }));
      parsedRequest = {
        ...parsedRequest,
        indices: [combinedPatterns.join(',')]
      };
    }
  }
  const {
    canWriteEventFilters,
    canReadEndpointList,
    canWriteTrustedApplications
  } = await context.getEndpointAuthz(deps.request);
  if (!canWriteEventFilters && parsedRequest.indices[0] === _constants.eventsIndexPattern || isTAAdvancedModeFeatureFlagEnabled && !canWriteTrustedApplications && parsedRequest.indices[0] === _constants.eventsIndexPattern || !canReadEndpointList && parsedRequest.indices[0] === _constants.METADATA_UNITED_INDEX) {
    throw new _errors.EndpointAuthorizationError();
  }
  return (0, _index_fields.requestIndexFieldSearch)(parsedRequest, deps, beatFields, indexPatterns, true);
};
exports.requestEndpointFieldsSearch = requestEndpointFieldsSearch;