"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationsDataPrebuiltRulesClient = void 0;
var _prebuilt_rule_assets_client = require("../../../detection_engine/prebuilt_rules/logic/rule_assets/prebuilt_rule_assets_client");
var _prebuilt_rule_objects_client = require("../../../detection_engine/prebuilt_rules/logic/rule_objects/prebuilt_rule_objects_client");
var _fetch_rule_versions_triad = require("../../../detection_engine/prebuilt_rules/logic/rule_versions/fetch_rule_versions_triad");
var _siem_migrations_data_base_client = require("../../common/data/siem_migrations_data_base_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* The minimum score required for a prebuilt rule to be considered correct */
const MIN_SCORE = 40;
/* The number of prebuilt rules the RAG will return, sorted by score */
const RETURNED_RULES = 5;

/* BULK_MAX_SIZE defines the number to break down the bulk operations by.
 * The 500 number was chosen as a reasonable number to avoid large payloads. It can be adjusted if needed.
 */
const BULK_MAX_SIZE = 500;
class RuleMigrationsDataPrebuiltRulesClient extends _siem_migrations_data_base_client.SiemMigrationsDataBaseClient {
  async getRuleVersionsMap() {
    const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(this.dependencies.savedObjectsClient);
    const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(this.dependencies.rulesClient);
    return (0, _fetch_rule_versions_triad.fetchRuleVersionsTriad)({
      ruleAssetsClient,
      ruleObjectsClient
    });
  }

  /** Indexes an array of prebuilt rules to be used with ELSER semantic search queries */
  async populate(ruleVersionsMap) {
    const filteredRules = [];
    ruleVersionsMap.forEach(ruleVersions => {
      const rule = ruleVersions.target;
      if (rule) {
        var _rule$threat;
        const mitreAttackIds = rule === null || rule === void 0 ? void 0 : (_rule$threat = rule.threat) === null || _rule$threat === void 0 ? void 0 : _rule$threat.flatMap(({
          technique
        }) => {
          var _technique$map;
          return (_technique$map = technique === null || technique === void 0 ? void 0 : technique.map(({
            id
          }) => id)) !== null && _technique$map !== void 0 ? _technique$map : [];
        });
        filteredRules.push({
          rule_id: rule.rule_id,
          name: rule.name,
          description: rule.description,
          elser_embedding: `${rule.name} - ${rule.description}`,
          ...((mitreAttackIds === null || mitreAttackIds === void 0 ? void 0 : mitreAttackIds.length) && {
            mitre_attack_ids: mitreAttackIds
          })
        });
      }
    });
    const index = await this.getIndexName();
    const createdAt = new Date().toISOString();
    let prebuiltRuleSlice;
    while ((prebuiltRuleSlice = filteredRules.splice(0, BULK_MAX_SIZE)).length) {
      await this.esClient.bulk({
        refresh: 'wait_for',
        operations: prebuiltRuleSlice.flatMap(prebuiltRule => [{
          update: {
            _index: index,
            _id: prebuiltRule.rule_id
          }
        }, {
          doc: {
            ...prebuiltRule,
            '@timestamp': createdAt
          },
          doc_as_upsert: true
        }])
      }, {
        requestTimeout: 10 * 60 * 1000
      } // 10 minutes
      ).then(response => {
        if (response.errors) {
          var _response$items$find, _response$items$find$, _response$items$find$2;
          // use the first error to throw
          const reason = (_response$items$find = response.items.find(item => {
            var _item$update;
            return (_item$update = item.update) === null || _item$update === void 0 ? void 0 : _item$update.error;
          })) === null || _response$items$find === void 0 ? void 0 : (_response$items$find$ = _response$items$find.update) === null || _response$items$find$ === void 0 ? void 0 : (_response$items$find$2 = _response$items$find$.error) === null || _response$items$find$2 === void 0 ? void 0 : _response$items$find$2.reason;
          throw new Error(reason !== null && reason !== void 0 ? reason : 'Unknown error');
        }
      }).catch(error => {
        this.logger.error(`Error indexing prebuilt rules embeddings: ${error.message}`);
        throw error;
      });
    }
  }

  /** Based on a LLM generated semantic string, returns the 5 best results with a score above 40 */
  async search(semanticString, techniqueIds) {
    const index = await this.getIndexName();
    const query = {
      bool: {
        should: [{
          semantic: {
            query: semanticString,
            field: 'elser_embedding',
            boost: 1.5
          }
        }, {
          multi_match: {
            query: semanticString,
            fields: ['name^2', 'description'],
            boost: 3
          }
        }, {
          multi_match: {
            query: techniqueIds,
            fields: ['mitre_attack_ids'],
            boost: 2
          }
        }]
      }
    };
    const results = await this.esClient.search({
      index,
      query,
      size: RETURNED_RULES,
      min_score: MIN_SCORE
    }).then(response => this.processResponseHits(response)).catch(error => {
      this.logger.error(`Error querying prebuilt rule details for ELSER: ${error.message}`);
      throw error;
    });
    return results;
  }
}
exports.RuleMigrationsDataPrebuiltRulesClient = RuleMigrationsDataPrebuiltRulesClient;