"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installTranslated = void 0;
var _error_helpers = require("../../../../../utils/error_helpers");
var _promise_pool = require("../../../../../utils/promise_pool");
var _tags = require("../../../common/api/util/tags");
var _tag_utils = require("./tag_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_DASHBOARDS_TO_CREATE_IN_PARALLEL = 10;
const installTranslated = async ({
  migrationId,
  ids,
  savedObjectsClient,
  securitySolutionContext,
  spaceId
}) => {
  const dashboardMigrationsClient = securitySolutionContext.siemMigrations.getDashboardsClient();
  let installedCount = 0;
  const installationErrors = [];

  // Get installable dashboard migrations
  const dashboardBatches = dashboardMigrationsClient.data.items.searchBatches(migrationId, {
    filters: {
      ids,
      installable: true
    }
  });
  let dashboardsToInstall = await dashboardBatches.next();
  while (dashboardsToInstall.length) {
    const {
      dashboardsToUpdate,
      errors
    } = await installDashboards(dashboardsToInstall, savedObjectsClient, spaceId);
    installedCount += dashboardsToUpdate.length;
    installationErrors.push(...errors);
    await dashboardMigrationsClient.data.items.update(dashboardsToUpdate);
    dashboardsToInstall = await dashboardBatches.next();
  }

  // Throw an error if needed
  if (installationErrors.length) {
    throw new Error(installationErrors.map(err => err.message).join(', '));
  }
  return installedCount;
};
exports.installTranslated = installTranslated;
const installDashboards = async (dashboardsToInstall, savedObjectsClient, spaceId) => {
  const errors = [];
  const dashboardsToUpdate = [];
  const createDashboardsOutcome = await (0, _promise_pool.initPromisePool)({
    concurrency: MAX_DASHBOARDS_TO_CREATE_IN_PARALLEL,
    items: dashboardsToInstall,
    executor: async dashboard => {
      try {
        var _dashboard$elastic_da;
        // Check if dashboard data exists
        if (!((_dashboard$elastic_da = dashboard.elastic_dashboard) !== null && _dashboard$elastic_da !== void 0 && _dashboard$elastic_da.data)) {
          throw new Error('Invalid Dashboard: No data to install');
        }

        // Parse the dashboard data (assuming it's JSON)
        const dashboardData = JSON.parse(dashboard.elastic_dashboard.data);
        const tagNames = [(0, _tags.getVendorTag)(dashboard.original_dashboard.vendor)];

        // Find or create tag references
        const tagReferences = await (0, _tag_utils.findOrCreateTagReferences)(savedObjectsClient, tagNames);
        const result = await savedObjectsClient.create('dashboard', {
          title: dashboard.original_dashboard.title,
          description: dashboard.original_dashboard.description,
          // Add other dashboard attributes as needed
          ...dashboardData.attributes
        }, {
          id: dashboard.id,
          references: tagReferences,
          initialNamespaces: [spaceId]
        });
        dashboardsToUpdate.push({
          id: dashboard.id,
          elastic_dashboard: {
            id: result.id
          }
        });
      } catch (error) {
        errors.push(new Error(`Error installing dashboard ${dashboard.id}: ${(0, _error_helpers.getErrorMessage)(error)}`));
      }
    }
  });
  errors.push(...createDashboardsOutcome.errors.map(err => new Error(`Error installing dashboard: ${(0, _error_helpers.getErrorMessage)(err)}`)));
  return {
    dashboardsToUpdate,
    errors
  };
};