"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationsDataItemClient = void 0;
var _constants = require("../../../../../common/siem_migrations/constants");
var _siem_migrations_data_base_client = require("./siem_migrations_data_base_client");
var _constants2 = require("./constants");
var _dsl_queries = require("./dsl_queries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* BULK_MAX_SIZE defines the number to break down the bulk operations by.
 * The 500 number was chosen as a reasonable number to avoid large payloads. It can be adjusted if needed. */
const BULK_MAX_SIZE = 500;
/* DEFAULT_SEARCH_BATCH_SIZE defines the default number of documents to retrieve per search operation
 * when retrieving search results in batches. */
const DEFAULT_SEARCH_BATCH_SIZE = 500;
class SiemMigrationsDataItemClient extends _siem_migrations_data_base_client.SiemMigrationsDataBaseClient {
  /** Indexes an array of migration items in pending status */
  async create(items) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const createdAt = new Date().toISOString();
    const batches = [];
    for (let i = 0; i < items.length; i += BULK_MAX_SIZE) {
      batches.push(items.slice(i, i + BULK_MAX_SIZE));
    }
    await Promise.all(batches.map(itemsSlice => this.esClient.bulk({
      refresh: 'wait_for',
      operations: itemsSlice.flatMap(item => [{
        create: {
          _index: index
        }
      }, {
        ...item,
        '@timestamp': createdAt,
        status: _constants.SiemMigrationStatus.PENDING,
        created_by: profileId,
        updated_by: profileId,
        updated_at: createdAt
      }])
    }).catch(error => {
      this.logger.error(`Error creating migration ${this.type}: ${error.message}`);
      throw error;
    })));
  }

  /** Updates an array of migration items */
  async update(itemsUpdate) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const updatedAt = new Date().toISOString();
    const batches = [];
    for (let i = 0; i < itemsUpdate.length; i += BULK_MAX_SIZE) {
      batches.push(itemsUpdate.slice(i, i + BULK_MAX_SIZE));
    }
    const batchPromises = batches.map(itemsUpdateSlice => this.esClient.bulk({
      refresh: 'wait_for',
      operations: itemsUpdateSlice.flatMap(item => {
        const {
          id,
          ...rest
        } = item;
        return [{
          update: {
            _index: index,
            _id: id
          }
        }, {
          doc: {
            ...rest,
            updated_by: profileId,
            updated_at: updatedAt
          }
        }];
      })
    }).catch(error => {
      this.logger.error(`Error updating migration ${this.type}: ${error.message}`);
      throw error;
    }));
    await Promise.all(batchPromises);
  }

  /** Retrieves an array of migration items of a specific migration */
  async get(migrationId, {
    filters = {},
    sort: sortParam = {},
    from,
    size
  } = {}) {
    const index = await this.getIndexName();
    const query = this.getFilterQuery(migrationId, filters);
    const sort = this.getSortOptions(sortParam);
    const result = await this.esClient.search({
      index,
      query,
      sort,
      from,
      size
    }).catch(error => {
      this.logger.error(`Error searching migration ${this.type}: ${error.message}`);
      throw error;
    });
    return {
      total: this.getTotalHits(result),
      data: this.processResponseHits(result)
    };
  }
  async getByQuery(migrationId, {
    queryDSL,
    from,
    size
  }) {
    const index = await this.getIndexName();
    const baseQuery = this.getFilterQuery(migrationId, {});
    const combinedQuery = {
      bool: {
        must: [baseQuery, queryDSL]
      }
    };
    const result = await this.esClient.search({
      index,
      query: combinedQuery,
      from,
      size
    }).catch(error => {
      this.logger.error(`Error searching migration ${this.type} by query: ${error.message}`);
      throw error;
    });
    return {
      total: this.getTotalHits(result),
      data: this.processResponseHits(result)
    };
  }

  /** Prepares bulk ES delete operations for the migration items based on migrationId. */
  async prepareDelete(migrationId) {
    const index = await this.getIndexName();
    const itemsToBeDeleted = await this.get(migrationId, {
      size: _constants2.MAX_ES_SEARCH_SIZE
    });
    const itemsToBeDeletedDocIds = itemsToBeDeleted.data.map(item => item.id);
    return itemsToBeDeletedDocIds.map(docId => ({
      delete: {
        _index: index,
        _id: docId
      }
    }));
  }

  /** Returns batching functions to traverse all the migration items search results */
  searchBatches(migrationId, options = {}) {
    const {
      size = DEFAULT_SEARCH_BATCH_SIZE,
      filters = {},
      scroll
    } = options;
    const query = this.getFilterQuery(migrationId, filters);
    const search = {
      query,
      sort: '_doc',
      scroll,
      size
    }; // sort by _doc to ensure consistent order
    try {
      return this.getSearchBatches(search);
    } catch (error) {
      this.logger.error(`Error scrolling migration ${this.type}: ${error.message}`);
      throw error;
    }
  }

  /** Retrieves the stats for the migrations items with the provided id */
  async getStats(migrationId) {
    var _result$aggregations, _value_as_string, _aggs$createdAt, _value_as_string2, _aggs$lastUpdatedAt;
    const index = await this.getIndexName();
    const query = this.getFilterQuery(migrationId);
    const aggregations = {
      status: {
        terms: {
          field: 'status'
        }
      },
      createdAt: {
        min: {
          field: '@timestamp'
        }
      },
      lastUpdatedAt: {
        max: {
          field: 'updated_at'
        }
      }
    };
    const result = await this.esClient.search({
      index,
      query,
      aggregations,
      _source: false
    }).catch(error => {
      this.logger.error(`Error getting migration ${this.type} stats: ${error.message}`);
      throw error;
    });
    const aggs = (_result$aggregations = result.aggregations) !== null && _result$aggregations !== void 0 ? _result$aggregations : {};
    return {
      id: migrationId,
      items: {
        total: this.getTotalHits(result),
        ...this.statusAggCounts(aggs.status)
      },
      created_at: (_value_as_string = (_aggs$createdAt = aggs.createdAt) === null || _aggs$createdAt === void 0 ? void 0 : _aggs$createdAt.value_as_string) !== null && _value_as_string !== void 0 ? _value_as_string : '',
      last_updated_at: (_value_as_string2 = (_aggs$lastUpdatedAt = aggs.lastUpdatedAt) === null || _aggs$lastUpdatedAt === void 0 ? void 0 : _aggs$lastUpdatedAt.value_as_string) !== null && _value_as_string2 !== void 0 ? _value_as_string2 : ''
    };
  }

  /** Retrieves the stats for all the migration items aggregated by migration id, in creation order */
  async getAllStats() {
    var _result$aggregations2, _ref;
    const index = await this.getIndexName();
    const aggregations = {
      migrationIds: {
        terms: {
          field: 'migration_id',
          order: {
            createdAt: 'asc'
          },
          size: _constants2.MAX_ES_SEARCH_SIZE
        },
        aggregations: {
          status: {
            terms: {
              field: 'status'
            }
          },
          createdAt: {
            min: {
              field: '@timestamp'
            }
          },
          lastUpdatedAt: {
            max: {
              field: 'updated_at'
            }
          }
        }
      }
    };
    const result = await this.esClient.search({
      index,
      aggregations,
      _source: false
    }).catch(error => {
      this.logger.error(`Error getting all migration ${this.type} stats: ${error.message}`);
      throw error;
    });
    const migrationsAgg = (_result$aggregations2 = result.aggregations) === null || _result$aggregations2 === void 0 ? void 0 : _result$aggregations2.migrationIds;
    const buckets = (_ref = migrationsAgg === null || migrationsAgg === void 0 ? void 0 : migrationsAgg.buckets) !== null && _ref !== void 0 ? _ref : [];
    return buckets.map(bucket => {
      var _bucket$createdAt, _bucket$lastUpdatedAt;
      return {
        id: `${bucket.key}`,
        items: {
          total: bucket.doc_count,
          ...this.statusAggCounts(bucket.status)
        },
        created_at: (_bucket$createdAt = bucket.createdAt) === null || _bucket$createdAt === void 0 ? void 0 : _bucket$createdAt.value_as_string,
        last_updated_at: (_bucket$lastUpdatedAt = bucket.lastUpdatedAt) === null || _bucket$lastUpdatedAt === void 0 ? void 0 : _bucket$lastUpdatedAt.value_as_string
      };
    });
  }

  /** Updates one migration item status to `processing` */
  async saveProcessing(id) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const doc = {
      status: _constants.SiemMigrationStatus.PROCESSING,
      updated_by: profileId,
      updated_at: new Date().toISOString()
    };
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for'
    }).catch(error => {
      this.logger.error(`Error updating migration ${this.type} status to processing: ${error.message}`);
      throw error;
    });
  }

  /** Updates one migration item with the provided data and sets the status to `completed` */
  async saveCompleted({
    id,
    ...migrationItem
  }) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const doc = {
      ...migrationItem,
      status: _constants.SiemMigrationStatus.COMPLETED,
      updated_by: profileId,
      updated_at: new Date().toISOString()
    };
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for'
    }).catch(error => {
      this.logger.error(`Error updating migration ${this.type} status to completed: ${error.message}`);
      throw error;
    });
  }

  /** Updates one migration item with the provided data and sets the status to `failed` */
  async saveError({
    id,
    ...migrationItem
  }) {
    const index = await this.getIndexName();
    const profileId = await this.getProfileUid();
    const doc = {
      ...migrationItem,
      status: _constants.SiemMigrationStatus.FAILED,
      updated_by: profileId,
      updated_at: new Date().toISOString()
    };
    await this.esClient.update({
      index,
      id,
      doc,
      refresh: 'wait_for'
    }).catch(error => {
      this.logger.error(`Error updating migration ${this.type} status to failed: ${error.message}`);
      throw error;
    });
  }

  /** Updates all the migration items with the provided id with status `processing` back to `pending` */
  async releaseProcessing(migrationId) {
    return this.updateStatus(migrationId, {
      status: _constants.SiemMigrationStatus.PROCESSING
    }, _constants.SiemMigrationStatus.PENDING);
  }

  /** Updates all the migration items with the provided id and with status `statusToQuery` to `statusToUpdate` */
  async updateStatus(migrationId, filter, statusToUpdate, {
    refresh = false
  } = {}) {
    const index = await this.getIndexName();
    const query = this.getFilterQuery(migrationId, filter);
    const script = {
      source: `ctx._source['status'] = '${statusToUpdate}'`
    };
    await this.esClient.updateByQuery({
      index,
      query,
      script,
      refresh
    }).catch(error => {
      this.logger.error(`Error updating migration ${this.type} status: ${error.message}`);
      throw error;
    });
  }
  statusAggCounts(statusAgg) {
    var _buckets$find$doc_cou, _buckets$find, _buckets$find$doc_cou2, _buckets$find2, _buckets$find$doc_cou3, _buckets$find3, _buckets$find$doc_cou4, _buckets$find4;
    const buckets = statusAgg.buckets;
    return {
      [_constants.SiemMigrationStatus.PENDING]: (_buckets$find$doc_cou = (_buckets$find = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.PENDING)) === null || _buckets$find === void 0 ? void 0 : _buckets$find.doc_count) !== null && _buckets$find$doc_cou !== void 0 ? _buckets$find$doc_cou : 0,
      [_constants.SiemMigrationStatus.PROCESSING]: (_buckets$find$doc_cou2 = (_buckets$find2 = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.PROCESSING)) === null || _buckets$find2 === void 0 ? void 0 : _buckets$find2.doc_count) !== null && _buckets$find$doc_cou2 !== void 0 ? _buckets$find$doc_cou2 : 0,
      [_constants.SiemMigrationStatus.COMPLETED]: (_buckets$find$doc_cou3 = (_buckets$find3 = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.COMPLETED)) === null || _buckets$find3 === void 0 ? void 0 : _buckets$find3.doc_count) !== null && _buckets$find$doc_cou3 !== void 0 ? _buckets$find$doc_cou3 : 0,
      [_constants.SiemMigrationStatus.FAILED]: (_buckets$find$doc_cou4 = (_buckets$find4 = buckets.find(({
        key
      }) => key === _constants.SiemMigrationStatus.FAILED)) === null || _buckets$find4 === void 0 ? void 0 : _buckets$find4.doc_count) !== null && _buckets$find$doc_cou4 !== void 0 ? _buckets$find$doc_cou4 : 0
    };
  }
  translationResultAggCount(resultAgg) {
    var _buckets$find$doc_cou5, _buckets$find5, _buckets$find$doc_cou6, _buckets$find6, _buckets$find$doc_cou7, _buckets$find7;
    const buckets = resultAgg.buckets;
    return {
      [_constants.MigrationTranslationResult.FULL]: (_buckets$find$doc_cou5 = (_buckets$find5 = buckets.find(({
        key
      }) => key === _constants.MigrationTranslationResult.FULL)) === null || _buckets$find5 === void 0 ? void 0 : _buckets$find5.doc_count) !== null && _buckets$find$doc_cou5 !== void 0 ? _buckets$find$doc_cou5 : 0,
      [_constants.MigrationTranslationResult.PARTIAL]: (_buckets$find$doc_cou6 = (_buckets$find6 = buckets.find(({
        key
      }) => key === _constants.MigrationTranslationResult.PARTIAL)) === null || _buckets$find6 === void 0 ? void 0 : _buckets$find6.doc_count) !== null && _buckets$find$doc_cou6 !== void 0 ? _buckets$find$doc_cou6 : 0,
      [_constants.MigrationTranslationResult.UNTRANSLATABLE]: (_buckets$find$doc_cou7 = (_buckets$find7 = buckets.find(({
        key
      }) => key === _constants.MigrationTranslationResult.UNTRANSLATABLE)) === null || _buckets$find7 === void 0 ? void 0 : _buckets$find7.doc_count) !== null && _buckets$find$doc_cou7 !== void 0 ? _buckets$find$doc_cou7 : 0
    };
  }
  getFilterQuery(migrationId, filters = {}) {
    const filter = [{
      term: {
        migration_id: migrationId
      }
    }];
    if (filters.status) {
      if (Array.isArray(filters.status)) {
        filter.push({
          terms: {
            status: filters.status
          }
        });
      } else {
        filter.push({
          term: {
            status: filters.status
          }
        });
      }
    }
    if (filters.ids) {
      filter.push({
        terms: {
          _id: filters.ids
        }
      });
    }
    if (filters.failed != null) {
      filter.push(filters.failed ? _dsl_queries.dsl.isFailed() : _dsl_queries.dsl.isNotFailed());
    }
    if (filters.fullyTranslated != null) {
      filter.push(filters.fullyTranslated ? _dsl_queries.dsl.isFullyTranslated() : _dsl_queries.dsl.isNotFullyTranslated());
    }
    if (filters.partiallyTranslated != null) {
      filter.push(filters.partiallyTranslated ? _dsl_queries.dsl.isPartiallyTranslated() : _dsl_queries.dsl.isNotPartiallyTranslated());
    }
    if (filters.untranslatable != null) {
      filter.push(filters.untranslatable ? _dsl_queries.dsl.isUntranslatable() : _dsl_queries.dsl.isNotUntranslatable());
    }
    return {
      bool: {
        filter
      }
    };
  }
}
exports.SiemMigrationsDataItemClient = SiemMigrationsDataItemClient;