"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.wrapHits = void 0;
var _utils = require("../utils/utils");
var _transform_hit_to_alert = require("./utils/transform_hit_to_alert");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * wrapHits is responsible for turning source events into alerts. Since we copy the source data into the alert, we are
 * effectively "wrapping" the source hits by adding alert metadata to them in `kibana.alert.*` fields and
 * generating an _id for the alert,
 * @param sharedParams SecuritySharedParams passed in from the common security rule wrapper logic
 * @param events Source events to turn into alerts
 * @param buildReasonMessage Function to generate the reason message based on source data
 * @returns Alerts ready to index
 */
const wrapHits = (sharedParams, events, buildReasonMessage) => {
  const wrappedDocs = events.map(event => {
    const id = (0, _utils.generateId)(event._index,
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
    event._id, String(event._version), `${sharedParams.spaceId}:${sharedParams.completeRule.alertId}`);
    const baseAlert = (0, _transform_hit_to_alert.transformHitToAlert)({
      sharedParams,
      doc: event,
      applyOverrides: true,
      buildReasonMessage,
      alertUuid: id
    });
    return {
      _id: id,
      _index: '',
      _source: {
        ...baseAlert
      }
    };
  });
  return wrappedDocs.filter(doc => !doc._source['kibana.alert.ancestors'].some(ancestor => ancestor.rule === sharedParams.completeRule.alertId));
};
exports.wrapHits = wrapHits;