"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDetectionRulesClient = void 0;
var _keys = require("@kbn/security-solution-features/keys");
var _with_security_span = require("../../../../../utils/with_security_span");
var _prebuilt_rule_assets_client = require("../../../prebuilt_rules/logic/rule_assets/prebuilt_rule_assets_client");
var _create_rule = require("./methods/create_rule");
var _delete_rule = require("./methods/delete_rule");
var _import_rule = require("./methods/import_rule");
var _import_rules = require("./methods/import_rules");
var _patch_rule = require("./methods/patch_rule");
var _update_rule = require("./methods/update_rule");
var _upgrade_prebuilt_rule = require("./methods/upgrade_prebuilt_rule");
var _revert_prebuilt_rule = require("./methods/revert_prebuilt_rule");
var _constants = require("../../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createDetectionRulesClient = ({
  actionsClient,
  rulesClient,
  mlAuthz,
  savedObjectsClient,
  productFeaturesService,
  license
}) => {
  const prebuiltRuleAssetClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(savedObjectsClient);
  return {
    getRuleCustomizationStatus() {
      /**
       * The prebuilt rules customization feature is gated by the license level.
       *
       * The license level is verified against the minimum required level for
       * the feature (Enterprise). However, since Serverless always operates at
       * the Enterprise license level, we must also check if the feature is
       * enabled in the product features. In Serverless, for different tiers,
       * unavailable features are disabled.
       */
      const isRulesCustomizationEnabled = license.hasAtLeast(_constants.MINIMUM_RULE_CUSTOMIZATION_LICENSE) && productFeaturesService.isEnabled(_keys.ProductFeatureKey.prebuiltRuleCustomization);
      return {
        isRulesCustomizationEnabled
      };
    },
    async createCustomRule(args) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.createCustomRule', async () => {
        var _args$params$enabled;
        return (0, _create_rule.createRule)({
          actionsClient,
          rulesClient,
          rule: {
            ...args.params,
            // For backwards compatibility, we default to true if not provided.
            // The default enabled value is false for prebuilt rules, and true
            // for custom rules.
            enabled: (_args$params$enabled = args.params.enabled) !== null && _args$params$enabled !== void 0 ? _args$params$enabled : true,
            immutable: false
          },
          mlAuthz
        });
      });
    },
    async createPrebuiltRule(args) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.createPrebuiltRule', async () => {
        return (0, _create_rule.createRule)({
          actionsClient,
          rulesClient,
          rule: {
            ...args.params,
            immutable: true
          },
          mlAuthz
        });
      });
    },
    async updateRule({
      ruleUpdate
    }) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.updateRule', async () => {
        return (0, _update_rule.updateRule)({
          actionsClient,
          rulesClient,
          prebuiltRuleAssetClient,
          mlAuthz,
          ruleUpdate
        });
      });
    },
    async patchRule({
      rulePatch
    }) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.patchRule', async () => {
        return (0, _patch_rule.patchRule)({
          actionsClient,
          rulesClient,
          prebuiltRuleAssetClient,
          mlAuthz,
          rulePatch
        });
      });
    },
    async deleteRule({
      ruleId
    }) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.deleteRule', async () => {
        return (0, _delete_rule.deleteRule)({
          rulesClient,
          ruleId
        });
      });
    },
    async upgradePrebuiltRule({
      ruleAsset
    }) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.upgradePrebuiltRule', async () => {
        return (0, _upgrade_prebuilt_rule.upgradePrebuiltRule)({
          actionsClient,
          rulesClient,
          ruleAsset,
          mlAuthz,
          prebuiltRuleAssetClient
        });
      });
    },
    async revertPrebuiltRule({
      ruleAsset,
      existingRule
    }) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.revertPrebuiltRule', async () => {
        return (0, _revert_prebuilt_rule.revertPrebuiltRule)({
          actionsClient,
          rulesClient,
          ruleAsset,
          mlAuthz,
          prebuiltRuleAssetClient,
          existingRule
        });
      });
    },
    async importRule(args) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.importRule', async () => {
        return (0, _import_rule.importRule)({
          actionsClient,
          rulesClient,
          importRulePayload: args,
          mlAuthz,
          prebuiltRuleAssetClient
        });
      });
    },
    async importRules(args) {
      return (0, _with_security_span.withSecuritySpan)('DetectionRulesClient.importRules', async () => {
        return (0, _import_rules.importRules)({
          ...args,
          detectionRulesClient: this,
          savedObjectsClient
        });
      });
    }
  };
};
exports.createDetectionRulesClient = createDetectionRulesClient;