"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeErrorResponse = exports.buildBulkResponse = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _lodash = require("lodash");
var _rule_management = require("../../../../../../../common/api/detection_engine/rule_management");
var _internal_rule_to_api_response = require("../../../logic/detection_rules_client/converters/internal_rule_to_api_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_ERROR_MESSAGE_LENGTH = 1000;
const buildBulkResponse = (response, {
  bulkAction,
  isDryRun = false,
  errors = [],
  updated = [],
  created = [],
  deleted = [],
  skipped = []
}) => {
  const numSucceeded = updated.length + created.length + deleted.length;
  const numSkipped = skipped.length;
  const numFailed = errors.length;
  const summary = {
    failed: numFailed,
    succeeded: numSucceeded,
    skipped: numSkipped,
    total: numSucceeded + numFailed + numSkipped
  };

  // if response is for dry_run, empty lists of rules returned, as rules are not actually updated and stored within ES
  // thus, it's impossible to return reliably updated/duplicated/deleted rules
  const results = isDryRun ? {
    updated: [],
    created: [],
    deleted: [],
    skipped: []
  } : {
    updated: updated.map(rule => (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule)),
    created: created.map(rule => (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule)),
    deleted: deleted.map(rule => (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule)),
    skipped
  };
  if (numFailed > 0) {
    let message = summary.succeeded > 0 ? 'Bulk edit partially failed' : 'Bulk edit failed';
    if (bulkAction === _rule_management.BulkActionTypeEnum.run) {
      message = summary.succeeded > 0 ? 'Bulk manual rule run partially failed' : 'Bulk manual rule run failed';
    }
    return response.custom({
      headers: {
        'content-type': 'application/json'
      },
      body: {
        message,
        status_code: 500,
        attributes: {
          errors: normalizeErrorResponse(errors),
          results,
          summary
        }
      },
      statusCode: 500
    });
  }
  const responseBody = {
    success: true,
    rules_count: summary.total,
    attributes: {
      results,
      summary
    }
  };
  return response.ok({
    body: responseBody
  });
};
exports.buildBulkResponse = buildBulkResponse;
const normalizeErrorResponse = errors => {
  const errorsMap = new Map();
  errors.forEach(errorObj => {
    let message;
    let statusCode = 500;
    let errorCode;
    let rule;
    // transform different error types (PromisePoolError<string> | PromisePoolError<RuleAlertType> | BulkOperationError)
    // to one common used in NormalizedRuleError
    if ('rule' in errorObj) {
      rule = errorObj.rule;
      message = errorObj.message;
    } else {
      const {
        error,
        item
      } = errorObj;
      const transformedError = error instanceof Error ? (0, _securitysolutionEsUtils.transformError)(error) : {
        message: String(error),
        statusCode: 500
      };
      errorCode = error === null || error === void 0 ? void 0 : error.errorCode;
      message = transformedError.message;
      statusCode = transformedError.statusCode;
      // The promise pool item is either a rule ID string or a rule object. We have
      // string IDs when we fail to fetch rules. Rule objects come from other
      // situations when we found a rule but failed somewhere else.
      rule = typeof item === 'string' ? {
        id: item
      } : {
        id: item.id,
        name: item.name
      };
    }
    if (errorsMap.has(message)) {
      var _errorsMap$get;
      (_errorsMap$get = errorsMap.get(message)) === null || _errorsMap$get === void 0 ? void 0 : _errorsMap$get.rules.push(rule);
    } else {
      errorsMap.set(message, {
        message: (0, _lodash.truncate)(message, {
          length: MAX_ERROR_MESSAGE_LENGTH
        }),
        status_code: statusCode,
        err_code: errorCode,
        rules: [rule]
      });
    }
  });
  return Array.from(errorsMap, ([_, normalizedError]) => normalizedError);
};
exports.normalizeErrorResponse = normalizeErrorResponse;