"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertRuleToDiffable = void 0;
var _utility_types = require("../../../utility_types");
var _extract_building_block_object = require("./extractors/extract_building_block_object");
var _extract_rule_data_query = require("./extractors/extract_rule_data_query");
var _extract_rule_data_source = require("./extractors/extract_rule_data_source");
var _extract_rule_name_override_object = require("./extractors/extract_rule_name_override_object");
var _extract_rule_schedule = require("./extractors/extract_rule_schedule");
var _extract_timeline_template_reference = require("./extractors/extract_timeline_template_reference");
var _extract_timestamp_override_object = require("./extractors/extract_timestamp_override_object");
var _normalize_rule_threshold = require("./normalizers/normalize_rule_threshold");
var _normalize_rule_response = require("./normalize_rule_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Structures and groups together similar fields for diffing purposes.
 * Read more in the JSDoc description of DiffableRule.
 */
const convertRuleToDiffable = ruleResponse => {
  const rule = (0, _normalize_rule_response.normalizeRuleResponse)(ruleResponse);
  const commonFields = extractDiffableCommonFields(rule);
  switch (rule.type) {
    case 'query':
      return {
        ...commonFields,
        ...extractDiffableCustomQueryFields(rule)
      };
    case 'saved_query':
      return {
        ...commonFields,
        ...extractDiffableSavedQueryFieldsFromRuleObject(rule)
      };
    case 'eql':
      return {
        ...commonFields,
        ...extractDiffableEqlFieldsFromRuleObject(rule)
      };
    case 'threat_match':
      return {
        ...commonFields,
        ...extractDiffableThreatMatchFieldsFromRuleObject(rule)
      };
    case 'threshold':
      return {
        ...commonFields,
        ...extractDiffableThresholdFieldsFromRuleObject(rule)
      };
    case 'machine_learning':
      return {
        ...commonFields,
        ...extractDiffableMachineLearningFieldsFromRuleObject(rule)
      };
    case 'new_terms':
      return {
        ...commonFields,
        ...extractDiffableNewTermsFieldsFromRuleObject(rule)
      };
    case 'esql':
      return {
        ...commonFields,
        ...extractDiffableEsqlFieldsFromRuleObject(rule)
      };
    default:
      return (0, _utility_types.assertUnreachable)(rule, 'Unhandled rule type');
  }
};
exports.convertRuleToDiffable = convertRuleToDiffable;
const extractDiffableCommonFields = rule => {
  var _rule$note, _rule$setup;
  return {
    // --------------------- REQUIRED FIELDS
    // Technical fields
    rule_id: rule.rule_id,
    version: rule.version,
    // Main domain fields
    name: rule.name,
    tags: rule.tags,
    description: rule.description,
    severity: rule.severity,
    severity_mapping: rule.severity_mapping,
    risk_score: rule.risk_score,
    risk_score_mapping: rule.risk_score_mapping,
    // About -> Advanced settings
    references: rule.references,
    false_positives: rule.false_positives,
    threat: rule.threat,
    note: (_rule$note = rule.note) !== null && _rule$note !== void 0 ? _rule$note : '',
    setup: (_rule$setup = rule.setup) !== null && _rule$setup !== void 0 ? _rule$setup : '',
    related_integrations: rule.related_integrations,
    required_fields: rule.required_fields,
    // Other domain fields
    rule_schedule: (0, _extract_rule_schedule.extractRuleSchedule)(rule),
    max_signals: rule.max_signals,
    // --------------------- OPTIONAL FIELDS
    investigation_fields: rule.investigation_fields,
    rule_name_override: (0, _extract_rule_name_override_object.extractRuleNameOverrideObject)(rule),
    timestamp_override: (0, _extract_timestamp_override_object.extractTimestampOverrideObject)(rule),
    timeline_template: (0, _extract_timeline_template_reference.extractTimelineTemplateReference)(rule),
    building_block: (0, _extract_building_block_object.extractBuildingBlockObject)(rule)
  };
};
const extractDiffableCustomQueryFields = rule => {
  return {
    type: rule.type,
    kql_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableSavedQueryFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    kql_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableEqlFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    eql_query: (0, _extract_rule_data_query.extractRuleEqlQuery)({
      query: rule.query,
      language: rule.language,
      filters: rule.filters,
      eventCategoryOverride: rule.event_category_override,
      timestampField: rule.timestamp_field,
      tiebreakerField: rule.tiebreaker_field
    }),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableEsqlFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    esql_query: (0, _extract_rule_data_query.extractRuleEsqlQuery)(rule.query, rule.language),
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableThreatMatchFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    kql_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    threat_query: (0, _extract_rule_data_query.extractInlineKqlQuery)(rule.threat_query, rule.threat_language, rule.threat_filters),
    threat_index: rule.threat_index,
    threat_mapping: rule.threat_mapping,
    threat_indicator_path: rule.threat_indicator_path,
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableThresholdFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    kql_query: (0, _extract_rule_data_query.extractRuleKqlQuery)(rule.query, rule.language, rule.filters, rule.saved_id),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    threshold: (0, _normalize_rule_threshold.normalizeRuleThreshold)(rule.threshold),
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableMachineLearningFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    machine_learning_job_id: rule.machine_learning_job_id,
    anomaly_threshold: rule.anomaly_threshold,
    alert_suppression: rule.alert_suppression
  };
};
const extractDiffableNewTermsFieldsFromRuleObject = rule => {
  return {
    type: rule.type,
    kql_query: (0, _extract_rule_data_query.extractInlineKqlQuery)(rule.query, rule.language, rule.filters),
    data_source: (0, _extract_rule_data_source.extractRuleDataSource)(rule.index, rule.data_view_id),
    new_terms_fields: rule.new_terms_fields,
    history_window_start: rule.history_window_start,
    alert_suppression: rule.alert_suppression
  };
};