"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleInstallSampleDataTask = exports.registerInstallSampleDataTaskDefinition = exports.getInstallTaskId = exports.INSTALL_SAMPLE_DATA_TASK_TYPE = void 0;
var _server = require("@kbn/task-manager-plugin/server");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INSTALL_SAMPLE_DATA_TASK_TYPE = exports.INSTALL_SAMPLE_DATA_TASK_TYPE = 'SampleDataIngest:InstallSampleData';
const getInstallTaskId = sampleType => {
  return `SampleDataIngest:Install:${sampleType}`;
};
exports.getInstallTaskId = getInstallTaskId;
const registerInstallSampleDataTaskDefinition = ({
  getServices,
  taskManager,
  core
}) => {
  taskManager.registerTaskDefinitions({
    [INSTALL_SAMPLE_DATA_TASK_TYPE]: {
      title: `Install sample data ${INSTALL_SAMPLE_DATA_TASK_TYPE}`,
      timeout: '10m',
      maxAttempts: 3,
      createTaskRunner: ({
        taskInstance,
        abortController
      }) => {
        var _taskInstance$params;
        const sampleType = taskInstance === null || taskInstance === void 0 ? void 0 : (_taskInstance$params = taskInstance.params) === null || _taskInstance$params === void 0 ? void 0 : _taskInstance$params.sampleType;
        return {
          async run() {
            const {
              sampleDataManager
            } = getServices();
            if (!sampleType) {
              throw new Error('Sample type is required');
            }
            try {
              const [coreStart] = await core.getStartServices();
              const esClient = coreStart.elasticsearch.client.asInternalUser;
              const soClient = coreStart.savedObjects.createInternalRepository();
              const soImporter = coreStart.savedObjects.createImporter(soClient);
              await sampleDataManager.installSampleData({
                sampleType,
                esClient,
                soClient,
                soImporter,
                abortController
              });
              return {
                state: {
                  status: 'completed'
                },
                shouldDeleteTask: true
              };
            } catch (error) {
              const normalizedError = error instanceof Error ? error : new Error(String(error !== null && error !== void 0 ? error : 'Unknown error'));
              const taskError = (0, _server.createTaskRunError)(normalizedError, _server.TaskErrorSource.FRAMEWORK);
              return {
                state: {
                  status: 'error',
                  errorMessage: normalizedError.message
                },
                error: taskError
              };
            }
          }
        };
      },
      stateSchemaByVersion: {}
    }
  });
};
exports.registerInstallSampleDataTaskDefinition = registerInstallSampleDataTaskDefinition;
const scheduleInstallSampleDataTask = async ({
  taskManager,
  logger,
  sampleType
}) => {
  const taskId = getInstallTaskId(sampleType);
  try {
    const initialState = {
      status: 'pending'
    };
    await taskManager.ensureScheduled({
      id: taskId,
      taskType: INSTALL_SAMPLE_DATA_TASK_TYPE,
      params: {
        sampleType
      },
      state: initialState,
      scope: ['sampleData']
    });
    await taskManager.runSoon(taskId);
    logger.info(`Task ${taskId} scheduled to run soon`);
  } catch (e) {
    if (!(0, _utils.isTaskCurrentlyRunningError)(e)) {
      throw e;
    }
  }
  return taskId;
};
exports.scheduleInstallSampleDataTask = scheduleInstallSampleDataTask;