"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.reindexServiceFactory = void 0;
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _upgradeAssistantPkgServer = require("@kbn/upgrade-assistant-pkg-server");
var _upgradeAssistantPkgCommon = require("@kbn/upgrade-assistant-pkg-common");
var _common = require("../../../common");
var _error = require("./error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const reindexServiceFactory = (esClient, actions, log, licensing, kibanaVersion) => {
  // ------ Utility functions
  const cleanupChanges = async reindexOp => {
    // Cancel reindex task if it was started but not completed
    if (reindexOp.attributes.lastCompletedStep === _common.ReindexStep.reindexStarted) {
      var _reindexOp$attributes;
      await esClient.tasks.cancel({
        task_id: (_reindexOp$attributes = reindexOp.attributes.reindexTaskId) !== null && _reindexOp$attributes !== void 0 ? _reindexOp$attributes : undefined
      }).catch(() => undefined); // Ignore any exceptions trying to cancel (it may have already completed).
    }

    // Set index back to writable if we ever got past this point.
    if (reindexOp.attributes.lastCompletedStep >= _common.ReindexStep.readonly) {
      await esClient.indices.putSettings({
        index: reindexOp.attributes.indexName,
        settings: {
          blocks: {
            write: false
          }
        }
      });
    }
    if (reindexOp.attributes.lastCompletedStep >= _common.ReindexStep.newIndexCreated && reindexOp.attributes.lastCompletedStep < _common.ReindexStep.aliasCreated) {
      await esClient.indices.delete({
        index: reindexOp.attributes.newIndexName
      });
    }
    return reindexOp;
  };

  // ------ Functions used to process the state machine

  /**
   * Sets a write-block on the original index. New data cannot be indexed until
   * the reindex is completed; there will be downtime for indexing until the
   * reindex is completed.
   * @param reindexOp
   */
  const setReadonly = async reindexOp => {
    const {
      indexName,
      rollupJob
    } = reindexOp.attributes;
    if (rollupJob) {
      await esClient.rollup.stopJob({
        id: rollupJob,
        wait_for_completion: true
      });
    }
    const putReadonly = await esClient.indices.putSettings({
      index: indexName,
      body: {
        blocks: {
          write: true
        }
      }
    });
    if (!putReadonly.acknowledged) {
      throw new Error(`Index could not be set to read-only.`);
    }
    return actions.updateReindexOp(reindexOp, {
      lastCompletedStep: _common.ReindexStep.readonly
    });
  };

  /**
   * Creates a new index with the same mappings and settings as the original index.
   * @param reindexOp
   */
  const createNewIndex = async reindexOp => {
    var _createIndex;
    const {
      indexName,
      newIndexName,
      settings: settingsStr
    } = reindexOp.attributes;
    const flatSettings = await actions.getFlatSettings(indexName);
    if (!flatSettings) {
      throw _error.error.indexNotFound(`Index ${indexName} does not exist.`);
    }
    let newSettings;
    try {
      newSettings = settingsStr ? JSON.parse(settingsStr) : {};
    } catch (err) {
      throw new Error(`Could not parse index settings: ${err}`);
    }
    const {
      settings = {}
    } = flatSettings;

    // Backup the current settings to restore them after the reindex
    // https://github.com/elastic/kibana/issues/201605
    const backupSettings = {
      'index.number_of_replicas': settings['index.number_of_replicas'],
      'index.refresh_interval': settings['index.refresh_interval']
    };

    // eslint-disable-next-line @typescript-eslint/naming-convention
    const settings_override = {
      ...newSettings,
      // Reindexing optimizations
      'index.number_of_replicas': 0,
      'index.refresh_interval': -1
    };
    let createIndex;
    try {
      createIndex = await esClient.transport.request({
        method: 'POST',
        path: `_create_from/${indexName}/${newIndexName}`,
        body: {
          settings_override
        }
      });
      /**
       * Response is expected to be:
       * {
       *   "acknowledged": true,
       *   "shards_acknowledged": true,
       *   "index": "test-copy"
       * }
       */
    } catch (err) {
      var _err$body, _err$body$error;
      // If for any reason the new index name generated by the `generateNewIndexName` already
      // exists (this could happen if kibana is restarted during reindexing), we can just go
      // ahead with the process without needing to create the index again.
      // See: https://github.com/elastic/kibana/issues/123816
      if ((err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) !== 'resource_already_exists_exception') {
        throw err;
      }
    }
    if (createIndex && !((_createIndex = createIndex) !== null && _createIndex !== void 0 && _createIndex.acknowledged)) {
      throw _error.error.cannotCreateIndex(`Index could not be created: ${newIndexName}`);
    }
    return actions.updateReindexOp(reindexOp, {
      lastCompletedStep: _common.ReindexStep.newIndexCreated,
      backupSettings
    });
  };

  /**
   * Begins the reindex process via Elasticsearch's Reindex API.
   * @param reindexOp
   */
  const startReindexing = async reindexOp => {
    const {
      indexName,
      reindexOptions
    } = reindexOp.attributes;

    // Where possible, derive reindex options at the last moment before reindexing
    // to prevent them from becoming stale as they wait in the queue.
    const indicesState = await (0, _upgradeAssistantPkgServer.esIndicesStateCheck)(esClient, [indexName]);
    const shouldOpenAndClose = indicesState[indexName] === 'closed';
    if (shouldOpenAndClose) {
      log.debug(`Detected closed index ${indexName}, opening...`);
      await esClient.indices.open({
        index: indexName
      });
    }
    const flatSettings = await actions.getFlatSettings(indexName);
    if (!flatSettings) {
      throw _error.error.indexNotFound(`Index ${indexName} does not exist.`);
    }
    const startReindexResponse = await esClient.reindex({
      refresh: true,
      wait_for_completion: false,
      source: {
        index: indexName
      },
      dest: {
        index: reindexOp.attributes.newIndexName
      },
      // Speed optimization https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-reindex.html#docs-reindex-automatic-slice
      // It doesn't work on CCS, but remote clusters should be upgraded individually with their own Upgrade Assistant.
      slices: 'auto'
    });
    return actions.updateReindexOp(reindexOp, {
      lastCompletedStep: _common.ReindexStep.reindexStarted,
      reindexTaskId: startReindexResponse.task === undefined ? startReindexResponse.task : String(startReindexResponse.task),
      reindexTaskPercComplete: 0,
      reindexOptions: {
        ...(reindexOptions !== null && reindexOptions !== void 0 ? reindexOptions : {}),
        // Indicate to downstream states whether we opened a closed index that should be
        // closed again.
        openAndClose: shouldOpenAndClose
      }
    });
  };

  /**
   * Polls Elasticsearch's Tasks API to see if the reindex operation has been completed.
   * @param reindexOp
   */
  const updateReindexStatus = async reindexOp => {
    var _taskResponse$task$st;
    const taskId = reindexOp.attributes.reindexTaskId;

    // Check reindexing task progress
    const taskResponse = await esClient.tasks.get({
      task_id: taskId,
      wait_for_completion: false
    });
    if (!taskResponse.completed) {
      // Updated the percent complete
      const perc = taskResponse.task.status.created / taskResponse.task.status.total;
      return actions.updateReindexOp(reindexOp, {
        reindexTaskPercComplete: perc
      });
    } else if (((_taskResponse$task$st = taskResponse.task.status) === null || _taskResponse$task$st === void 0 ? void 0 : _taskResponse$task$st.canceled) === 'by user request') {
      // Set the status to cancelled
      reindexOp = await actions.updateReindexOp(reindexOp, {
        status: _upgradeAssistantPkgCommon.ReindexStatus.cancelled
      });

      // Do any other cleanup work necessary
      reindexOp = await cleanupChanges(reindexOp);
    } else {
      var _taskResponse$respons, _taskResponse$respons2;
      // Check that no failures occurred
      if ((_taskResponse$respons = taskResponse.response) !== null && _taskResponse$respons !== void 0 && (_taskResponse$respons2 = _taskResponse$respons.failures) !== null && _taskResponse$respons2 !== void 0 && _taskResponse$respons2.length) {
        // Include the entire task result in the error message. This should be guaranteed
        // to be JSON-serializable since it just came back from Elasticsearch.
        throw _error.error.reindexTaskFailed(`Reindexing failed: ${JSON.stringify(taskResponse)}`);
      }

      // Update the status
      reindexOp = await actions.updateReindexOp(reindexOp, {
        lastCompletedStep: _common.ReindexStep.reindexCompleted,
        reindexTaskPercComplete: 1
      });
    }
    try {
      // Best effort, delete the task from ES .tasks index...
      await esClient.delete({
        index: '.tasks',
        id: taskId
      });
    } catch (e) {
      // We explicitly ignore authz related error codes bc we expect this to be
      // very common when deleting from .tasks
      if ((e === null || e === void 0 ? void 0 : e.statusCode) !== 401 && (e === null || e === void 0 ? void 0 : e.statusCode) !== 403) {
        log.warn(e);
      }
    }
    return reindexOp;
  };
  const getIndexInfo = async indexName => {
    var _response$indexName$a, _response$indexName, _response$indexName$s, _response$indexName2, _response$indexName2$, _response$indexName3;
    const response = await esClient.indices.get({
      index: indexName,
      features: ['aliases', 'settings']
    });
    const aliases = (_response$indexName$a = (_response$indexName = response[indexName]) === null || _response$indexName === void 0 ? void 0 : _response$indexName.aliases) !== null && _response$indexName$a !== void 0 ? _response$indexName$a : {};
    const settings = (_response$indexName$s = (_response$indexName2 = response[indexName]) === null || _response$indexName2 === void 0 ? void 0 : (_response$indexName2$ = _response$indexName2.settings) === null || _response$indexName2$ === void 0 ? void 0 : _response$indexName2$.index) !== null && _response$indexName$s !== void 0 ? _response$indexName$s : {};
    const isInDataStream = Boolean((_response$indexName3 = response[indexName]) === null || _response$indexName3 === void 0 ? void 0 : _response$indexName3.data_stream);

    // Check if the index is a follower index
    let isFollowerIndex = false;
    try {
      var _ccrResponse$follower;
      const ccrResponse = await esClient.ccr.followInfo({
        index: indexName
      });
      isFollowerIndex = ((_ccrResponse$follower = ccrResponse.follower_indices) === null || _ccrResponse$follower === void 0 ? void 0 : _ccrResponse$follower.length) > 0;
    } catch (err) {
      // If the API returns a 404, it means the index is not a follower index
      // Any other error should be ignored and we'll default to false
      isFollowerIndex = false;
    }
    return {
      aliases,
      settings,
      isInDataStream,
      isFollowerIndex
    };
  };
  const isIndexHidden = async indexName => {
    var _response$indexName4, _response$indexName4$, _response$indexName4$2;
    const response = await esClient.indices.getSettings({
      index: indexName
    });
    const isHidden = (_response$indexName4 = response[indexName]) === null || _response$indexName4 === void 0 ? void 0 : (_response$indexName4$ = _response$indexName4.settings) === null || _response$indexName4$ === void 0 ? void 0 : (_response$indexName4$2 = _response$indexName4$.index) === null || _response$indexName4$2 === void 0 ? void 0 : _response$indexName4$2.hidden;
    return isHidden === true || isHidden === 'true';
  };

  /**
   * Restores the original index settings in the new index that had other defaults for reindexing performance reasons
   * Also removes any deprecated index settings found in warnings
   * @param reindexOp
   */
  const restoreIndexSettings = async reindexOp => {
    var _indexSettingsWarning;
    const {
      newIndexName,
      backupSettings,
      indexName
    } = reindexOp.attributes;

    // Build settings to restore or remove
    const settingsToApply = {
      // Defaulting to null in case the original setting was empty to remove the setting.
      'index.number_of_replicas': null,
      'index.refresh_interval': null,
      ...backupSettings
    };

    // Get the warnings for this index to check for deprecated settings
    const flatSettings = await actions.getFlatSettings(indexName);
    const warnings = flatSettings ? (0, _upgradeAssistantPkgServer.getReindexWarnings)(flatSettings, kibanaVersion.getMajorVersion()) : undefined;
    const indexSettingsWarning = warnings === null || warnings === void 0 ? void 0 : warnings.find(warning => warning.warningType === 'indexSetting' && (warning.flow === 'reindex' || warning.flow === 'all'));

    // If there are deprecated settings, set them to null to remove them
    if (indexSettingsWarning !== null && indexSettingsWarning !== void 0 && (_indexSettingsWarning = indexSettingsWarning.meta) !== null && _indexSettingsWarning !== void 0 && _indexSettingsWarning.deprecatedSettings) {
      const deprecatedSettings = indexSettingsWarning.meta.deprecatedSettings;
      for (const setting of deprecatedSettings) {
        settingsToApply[setting] = null;
      }
      log.info(`Removing deprecated settings ${deprecatedSettings.join(', ')} from reindexed index ${newIndexName}`);
    }
    const settingsResponse = await esClient.indices.putSettings({
      index: newIndexName,
      settings: settingsToApply,
      // Any static settings that would ordinarily only be updated on closed indices
      // will be updated by automatically closing and reopening the affected indices.
      // @ts-ignore - This is not in the ES types, but it is a valid option
      reopen: true
    });
    if (!settingsResponse.acknowledged) {
      throw _error.error.cannotCreateIndex(`The original index settings could not be restored.`);
    }
    return actions.updateReindexOp(reindexOp, {
      lastCompletedStep: _common.ReindexStep.indexSettingsRestored
    });
  };

  /**
   * Creates an alias that points the old index to the new index, deletes the old index.
   * If old index was closed, the new index will also be closed.
   *
   * @note indexing/writing to the new index is effectively enabled after this action!
   * @param reindexOp
   */
  const switchAlias = async reindexOp => {
    var _reindexOp$attributes2;
    const {
      indexName,
      newIndexName,
      reindexOptions
    } = reindexOp.attributes;
    const existingAliases = (await getIndexInfo(indexName)).aliases;
    const extraAliases = Object.keys(existingAliases).map(aliasName => ({
      add: {
        index: newIndexName,
        alias: aliasName,
        ...existingAliases[aliasName]
      }
    }));
    const isHidden = await isIndexHidden(indexName);
    const updateAliasActions = (_reindexOp$attributes2 = reindexOp.attributes.reindexOptions) !== null && _reindexOp$attributes2 !== void 0 && _reindexOp$attributes2.deleteOldIndex ? [{
      add: {
        index: newIndexName,
        alias: indexName,
        is_hidden: isHidden
      }
    }, {
      remove_index: {
        index: indexName
      }
    }, ...extraAliases] : extraAliases;
    if (updateAliasActions.length) {
      const aliasResponse = await esClient.indices.updateAliases({
        actions: updateAliasActions
      });
      if (!aliasResponse.acknowledged) {
        throw _error.error.cannotCreateIndex(`Index aliases could not be created.`);
      }
    }
    if ((reindexOptions === null || reindexOptions === void 0 ? void 0 : reindexOptions.openAndClose) === true) {
      await esClient.indices.close({
        index: indexName
      });
    }
    if (reindexOp.attributes.rollupJob) {
      // start the rollup job. rollupJob is undefined if the rollup job is stopped
      await esClient.rollup.startJob({
        id: reindexOp.attributes.rollupJob
      });
    }
    return actions.updateReindexOp(reindexOp, {
      lastCompletedStep: _common.ReindexStep.aliasCreated
    });
  };

  // ------ The service itself

  return {
    async hasRequiredPrivileges(names) {
      /**
       * To avoid a circular dependency on Security we use a work around
       * here to detect whether Security is available and enabled
       * (i.e., via the licensing plugin). This enables Security to use
       * functionality exposed through Upgrade Assistant.
       */
      const license = await (0, _rxjs.firstValueFrom)(licensing.license$);
      const securityFeature = license.getFeature('security');

      // If security is disabled or unavailable, return true.
      if (!securityFeature || !(securityFeature.isAvailable && securityFeature.isEnabled)) {
        return true;
      }
      const resp = await esClient.security.hasPrivileges({
        cluster: ['manage'],
        index: [{
          names,
          allow_restricted_indices: true,
          privileges: ['all']
        }]
      });
      return resp.has_all_requested;
    },
    async detectReindexWarnings(indexName) {
      const flatSettings = await actions.getFlatSettings(indexName);
      if (!flatSettings) {
        return undefined;
      } else {
        return [
        // By default all reindexing operations will replace an index with an alias (with the same name)
        // pointing to a newly created "reindexed" index. This is destructive as delete operations originally
        // done on the index itself will now need to be done to the "reindexed-{indexName}"
        {
          warningType: 'makeIndexReadonly',
          flow: 'readonly'
        }, {
          warningType: 'replaceIndexWithAlias',
          flow: 'reindex'
        }, ...(0, _upgradeAssistantPkgServer.getReindexWarnings)(flatSettings, kibanaVersion.getMajorVersion())];
      }
    },
    async createReindexOperation({
      indexName,
      newIndexName,
      opts,
      settings
    }) {
      const indexExists = await esClient.indices.exists({
        index: indexName
      });
      if (!indexExists) {
        throw _error.error.indexNotFound(_i18n.i18n.translate('xpack.reindexService.error.indexNotFound', {
          defaultMessage: 'Index {indexName} does not exist in this cluster.',
          values: {
            indexName
          }
        }));
      }
      const newIndexExists = await esClient.indices.exists({
        index: newIndexName
      });
      if (newIndexExists) {
        throw _error.error.indexAlreadyExists(_i18n.i18n.translate('xpack.reindexService.error.indexAlreadyExists', {
          defaultMessage: 'The index {newIndexName} already exists. Please try again.',
          values: {
            newIndexName
          }
        }));
      }
      const existingReindexOps = await actions.findReindexOperations(indexName);
      if (existingReindexOps.total !== 0) {
        const existingOp = existingReindexOps.saved_objects[0];
        if (existingOp.attributes.status === _upgradeAssistantPkgCommon.ReindexStatus.failed || existingOp.attributes.status === _upgradeAssistantPkgCommon.ReindexStatus.cancelled || existingOp.attributes.status === _upgradeAssistantPkgCommon.ReindexStatus.completed) {
          // Delete the existing one if it failed or was cancelled to give a chance to retry.
          await actions.deleteReindexOp(existingOp);
        } else {
          throw _error.error.reindexAlreadyInProgress(`A reindex operation already in-progress for ${indexName}`);
        }
      }
      const reindexOptions = {};
      if (opts !== null && opts !== void 0 && opts.enqueue) {
        reindexOptions.queueSettings = {
          queuedAt: Date.now()
        };
      }
      if (opts !== null && opts !== void 0 && opts.deleteOldIndex) {
        reindexOptions.deleteOldIndex = true;
      }
      return actions.createReindexOp({
        indexName,
        newIndexName,
        reindexOptions,
        settings
      });
    },
    async findReindexOperation(indexName) {
      const findResponse = await actions.findReindexOperations(indexName);

      // Bail early if it does not exist or there is more than one.
      if (findResponse.total === 0) {
        return null;
      } else if (findResponse.total > 1) {
        throw _error.error.multipleReindexJobsFound(`More than one reindex operation found for ${indexName}`);
      }
      return findResponse.saved_objects[0];
    },
    async cleanupReindexOperations(indexNames) {
      const performCleanup = async indexName => {
        const existingReindexOps = await actions.findReindexOperations(indexName);
        if (existingReindexOps && existingReindexOps.total !== 0) {
          const existingOp = existingReindexOps.saved_objects[0];
          if (existingOp.attributes.status === _upgradeAssistantPkgCommon.ReindexStatus.completed) {
            // Delete the existing one if its status is completed, but still contains deprecation warnings
            // example scenario: index was upgraded, but then deleted and restored with an old snapshot
            await actions.deleteReindexOp(existingOp);
          }
        }
      };
      await Promise.all(indexNames.map(performCleanup));
    },
    findAllByStatus: actions.findAllByStatus,
    async processNextStep(reindexOp) {
      return actions.runWhileLocked(reindexOp, async lockedReindexOp => {
        try {
          switch (lockedReindexOp.attributes.lastCompletedStep) {
            case _common.ReindexStep.created:
              lockedReindexOp = await setReadonly(lockedReindexOp);
              break;
            case _common.ReindexStep.readonly:
              lockedReindexOp = await createNewIndex(lockedReindexOp);
              break;
            case _common.ReindexStep.newIndexCreated:
              lockedReindexOp = await startReindexing(lockedReindexOp);
              break;
            case _common.ReindexStep.reindexStarted:
              lockedReindexOp = await updateReindexStatus(lockedReindexOp);
              break;
            case _common.ReindexStep.reindexCompleted:
              lockedReindexOp = await restoreIndexSettings(lockedReindexOp);
              break;
            case _common.ReindexStep.indexSettingsRestored:
              lockedReindexOp = await switchAlias(lockedReindexOp);
              break;
            case _common.ReindexStep.aliasCreated:
              lockedReindexOp = await actions.updateReindexOp(lockedReindexOp, {
                status: _upgradeAssistantPkgCommon.ReindexStatus.completed
              });
              break;
            default:
              break;
          }
        } catch (e) {
          log.error(`Reindexing step failed: ${e instanceof Error ? e.stack : e.toString()}`);

          // Trap the exception and add the message to the object so the UI can display it.
          lockedReindexOp = await actions.updateReindexOp(lockedReindexOp, {
            status: _upgradeAssistantPkgCommon.ReindexStatus.failed,
            errorMessage: e.toString()
          });

          // Cleanup any changes, ignoring any errors.
          lockedReindexOp = await cleanupChanges(lockedReindexOp).catch(err => lockedReindexOp);
        }
        return lockedReindexOp;
      });
    },
    async pauseReindexOperation(indexName) {
      const reindexOp = await this.findReindexOperation(indexName);
      if (!reindexOp) {
        throw new Error(`No reindex operation found for index ${indexName}`);
      }
      return actions.runWhileLocked(reindexOp, async op => {
        if (op.attributes.status === _upgradeAssistantPkgCommon.ReindexStatus.paused) {
          // Another node already paused the operation, don't do anything
          return reindexOp;
        } else if (op.attributes.status !== _upgradeAssistantPkgCommon.ReindexStatus.inProgress) {
          throw new Error(`Reindex operation must be inProgress in order to be paused.`);
        }
        return actions.updateReindexOp(op, {
          status: _upgradeAssistantPkgCommon.ReindexStatus.paused
        });
      });
    },
    async resumeReindexOperation(indexName, opts) {
      const reindexOp = await this.findReindexOperation(indexName);
      if (!reindexOp) {
        throw new Error(`No reindex operation found for index ${indexName}`);
      }
      return actions.runWhileLocked(reindexOp, async op => {
        if (op.attributes.status === _upgradeAssistantPkgCommon.ReindexStatus.inProgress) {
          // Another node already resumed the operation, don't do anything
          return reindexOp;
        } else if (op.attributes.status !== _upgradeAssistantPkgCommon.ReindexStatus.paused) {
          throw new Error(`Reindex operation must be paused in order to be resumed.`);
        }
        const queueSettings = opts !== null && opts !== void 0 && opts.enqueue ? {
          queuedAt: Date.now()
        } : undefined;
        return actions.updateReindexOp(op, {
          status: _upgradeAssistantPkgCommon.ReindexStatus.inProgress,
          reindexOptions: queueSettings ? {
            queueSettings
          } : undefined
        });
      });
    },
    async startQueuedReindexOperation(indexName) {
      var _reindexOp$attributes3;
      const reindexOp = await this.findReindexOperation(indexName);
      if (!reindexOp) {
        throw _error.error.indexNotFound(`No reindex operation found for index ${indexName}`);
      }
      if (!((_reindexOp$attributes3 = reindexOp.attributes.reindexOptions) !== null && _reindexOp$attributes3 !== void 0 && _reindexOp$attributes3.queueSettings)) {
        throw _error.error.reindexIsNotInQueue(`Reindex operation ${indexName} is not in the queue.`);
      }
      return actions.runWhileLocked(reindexOp, async lockedReindexOp => {
        const {
          reindexOptions
        } = lockedReindexOp.attributes;
        reindexOptions.queueSettings.startedAt = Date.now();
        return actions.updateReindexOp(lockedReindexOp, {
          reindexOptions
        });
      });
    },
    async cancelReindexing(indexName) {
      const reindexOp = await this.findReindexOperation(indexName);
      if (!reindexOp) {
        throw _error.error.indexNotFound(`No reindex operation found for index ${indexName}`);
      } else if (reindexOp.attributes.status !== _upgradeAssistantPkgCommon.ReindexStatus.inProgress) {
        throw _error.error.reindexCannotBeCancelled(`Reindex operation is not in progress`);
      } else if (reindexOp.attributes.lastCompletedStep !== _common.ReindexStep.reindexStarted) {
        throw _error.error.reindexCannotBeCancelled(`Reindex operation is not currently waiting for reindex task to complete`);
      }
      const resp = await esClient.tasks.cancel({
        task_id: reindexOp.attributes.reindexTaskId
      });
      if (resp.node_failures && resp.node_failures.length > 0) {
        throw _error.error.reindexCannotBeCancelled(`Could not cancel reindex.`);
      }
      return reindexOp;
    },
    getIndexInfo
  };
};
exports.reindexServiceFactory = reindexServiceFactory;