"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseExperimentalConfigValue = exports.getExperimentalAllowedValues = exports.allowedExperimentalValues = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A list of allowed values that can be used in `xpack.osquery.enableExperimental`.
 * This object is then used to validate and parse the value entered.
 */
const allowedExperimentalValues = exports.allowedExperimentalValues = Object.freeze({
  /**
   * Placeholder feature flag for testing experimental features infrastructure.
   * This should be removed once the first real experimental feature is added.
   */
  _placeholder: false
});
const allowedKeys = Object.keys(allowedExperimentalValues);
const disableExperimentalPrefix = 'disable:';

/**
 * Parses the string value used in `xpack.osquery.enableExperimental` kibana configuration,
 * which should be an array of strings corresponding to allowedExperimentalValues keys.
 * Use the `disable:` prefix to disable a feature.
 *
 * @param configValue
 */
const parseExperimentalConfigValue = configValue => {
  const enabledFeatures = {};
  const invalidKeys = [];
  for (let value of configValue) {
    const isDisabled = value.startsWith(disableExperimentalPrefix);
    if (isDisabled) {
      value = value.replace(disableExperimentalPrefix, '');
    }
    if (!allowedKeys.includes(value)) {
      invalidKeys.push(value);
    } else {
      enabledFeatures[value] = !isDisabled;
    }
  }
  return {
    features: {
      ...allowedExperimentalValues,
      ...enabledFeatures
    },
    invalid: invalidKeys
  };
};
exports.parseExperimentalConfigValue = parseExperimentalConfigValue;
const getExperimentalAllowedValues = () => [...allowedKeys];
exports.getExperimentalAllowedValues = getExperimentalAllowedValues;