"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTelemetryCollector = registerTelemetryCollector;
var _query_utils = require("./query_utils");
var _usage_counters = require("./usage_counters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Telemetry payload schema for Agent Builder
 */

/**
 * Register telemetry collector for Agent Builder
 * @param usageCollection - Usage collection setup contract
 * @param logger - Logger instance
 */
function registerTelemetryCollector(usageCollection, logger) {
  if (!usageCollection) {
    logger.debug('Usage collection not available, skipping telemetry collector registration');
    return;
  }
  usageCollection.registerCollector(usageCollection.makeUsageCollector({
    type: 'agent_builder',
    isReady: () => true,
    schema: {
      custom_tools: {
        total: {
          type: 'long',
          _meta: {
            description: 'Total number of custom tools created by users'
          }
        },
        by_type: {
          type: 'array',
          items: {
            type: {
              type: 'keyword',
              _meta: {
                description: 'Tool type (esql, index_search, workflow, builtin)'
              }
            },
            count: {
              type: 'long',
              _meta: {
                description: 'Number of tools of this type'
              }
            }
          }
        }
      },
      custom_agents: {
        total: {
          type: 'long',
          _meta: {
            description: 'Total number of custom agents created by users'
          }
        }
      },
      conversations: {
        total: {
          type: 'long',
          _meta: {
            description: 'Total number of conversations'
          }
        },
        total_rounds: {
          type: 'long',
          _meta: {
            description: 'Total conversation rounds across all conversations'
          }
        },
        avg_rounds_per_conversation: {
          type: 'float',
          _meta: {
            description: 'Average rounds per conversation'
          }
        },
        rounds_distribution: {
          type: 'array',
          items: {
            bucket: {
              type: 'keyword',
              _meta: {
                description: 'Round count bucket (1-5, 6-10, 11-20, 21-50, 51+)'
              }
            },
            count: {
              type: 'long',
              _meta: {
                description: 'Number of conversations in this bucket'
              }
            }
          }
        },
        tokens_used: {
          type: 'long',
          _meta: {
            description: 'Total tokens used across all conversations (input + output)'
          }
        },
        average_tokens_per_conversation: {
          type: 'float',
          _meta: {
            description: 'Average tokens per conversation'
          }
        }
      },
      query_to_result_time: {
        p50: {
          type: 'long',
          _meta: {
            description: '50th percentile query-to-result time in milliseconds'
          }
        },
        p75: {
          type: 'long',
          _meta: {
            description: '75th percentile query-to-result time in milliseconds'
          }
        },
        p90: {
          type: 'long',
          _meta: {
            description: '90th percentile query-to-result time in milliseconds'
          }
        },
        p95: {
          type: 'long',
          _meta: {
            description: '95th percentile query-to-result time in milliseconds'
          }
        },
        p99: {
          type: 'long',
          _meta: {
            description: '99th percentile query-to-result time in milliseconds'
          }
        },
        mean: {
          type: 'long',
          _meta: {
            description: 'Mean query-to-result time in milliseconds'
          }
        }
      },
      tool_calls: {
        total: {
          type: 'long',
          _meta: {
            description: 'Total tool calls across all sources'
          }
        },
        by_source: {
          default_agent: {
            type: 'long',
            _meta: {
              description: 'Tool calls from default agents'
            }
          },
          custom_agent: {
            type: 'long',
            _meta: {
              description: 'Tool calls from custom agents'
            }
          },
          mcp: {
            type: 'long',
            _meta: {
              description: 'Tool calls from MCP clients'
            }
          },
          api: {
            type: 'long',
            _meta: {
              description: 'Direct tool calls via API'
            }
          },
          a2a: {
            type: 'long',
            _meta: {
              description: 'Tool calls from agent-to-agent communication'
            }
          }
        }
      },
      llm_usage: {
        by_provider: {
          type: 'array',
          items: {
            provider: {
              type: 'keyword',
              _meta: {
                description: 'LLM provider name (e.g., openai, bedrock)'
              }
            },
            count: {
              type: 'long',
              _meta: {
                description: 'Number of LLM invocations for this provider'
              }
            }
          }
        },
        by_model: {
          type: 'array',
          items: {
            model: {
              type: 'keyword',
              _meta: {
                description: 'LLM model identifier'
              }
            },
            count: {
              type: 'long',
              _meta: {
                description: 'Number of LLM invocations for this model'
              }
            }
          }
        }
      },
      errors: {
        total: {
          type: 'long',
          _meta: {
            description: 'Total number of errors surfaced to users'
          }
        },
        avg_errors_per_conversation: {
          type: 'float',
          _meta: {
            description: 'Average number of errors per conversation that had errors'
          }
        },
        total_conversations_with_errors: {
          type: 'long',
          _meta: {
            description: 'Total number of unique conversations that had at least one error'
          }
        },
        by_type: {
          type: 'array',
          items: {
            type: {
              type: 'keyword',
              _meta: {
                description: 'Error type/code (e.g., internalError, badRequest)'
              }
            },
            count: {
              type: 'long',
              _meta: {
                description: 'Number of errors of this type'
              }
            }
          }
        }
      }
    },
    fetch: async context => {
      const {
        esClient,
        soClient
      } = context;
      const queryUtils = new _query_utils.QueryUtils(esClient, soClient, logger);
      try {
        const customTools = await queryUtils.getCustomToolsMetrics();
        const customAgents = await queryUtils.getCustomAgentsMetrics();
        const conversations = await queryUtils.getConversationMetrics();
        const queryTimeCounters = await queryUtils.getCountersByPrefix(_usage_counters.ONECHAT_USAGE_DOMAIN, 'query_to_result_time_');
        const queryToResultTime = queryUtils.calculatePercentilesFromBuckets(queryTimeCounters);
        const toolCallCounters = await queryUtils.getCountersByPrefix(_usage_counters.ONECHAT_USAGE_DOMAIN, 'tool_call_');
        const toolCallsBySource = {
          default_agent: toolCallCounters.get('tool_call_default_agent') || 0,
          custom_agent: toolCallCounters.get('tool_call_custom_agent') || 0,
          mcp: toolCallCounters.get('tool_call_mcp') || 0,
          api: toolCallCounters.get('tool_call_api') || 0,
          a2a: toolCallCounters.get('tool_call_a2a') || 0
        };
        const totalToolCalls = Object.values(toolCallsBySource).reduce((sum, count) => sum + count, 0);
        const llmProviderCounters = await queryUtils.getCountersByPrefix(_usage_counters.ONECHAT_USAGE_DOMAIN, `${_usage_counters.ONECHAT_USAGE_DOMAIN}_llm_provider_`);
        const llmModelCounters = await queryUtils.getCountersByPrefix(_usage_counters.ONECHAT_USAGE_DOMAIN, `${_usage_counters.ONECHAT_USAGE_DOMAIN}_llm_model_`);
        const llmUsageByProvider = [];
        for (const [counterName, count] of Array.from(llmProviderCounters.entries())) {
          const provider = counterName.replace(`${_usage_counters.ONECHAT_USAGE_DOMAIN}_llm_provider_`, '');
          if (provider && count > 0) {
            llmUsageByProvider.push({
              provider,
              count
            });
          }
        }
        llmUsageByProvider.sort((a, b) => b.count - a.count);
        const llmUsageByModel = [];
        for (const [counterName, count] of Array.from(llmModelCounters.entries())) {
          const model = counterName.replace(`${_usage_counters.ONECHAT_USAGE_DOMAIN}_llm_model_`, '');
          if (model && count > 0) {
            llmUsageByModel.push({
              model,
              count
            });
          }
        }
        llmUsageByModel.sort((a, b) => b.count - a.count);
        const errorCounters = await queryUtils.getCountersByPrefix(_usage_counters.ONECHAT_USAGE_DOMAIN, 'error_');
        const totalErrors = errorCounters.get(`${_usage_counters.ONECHAT_USAGE_DOMAIN}_error_total`) || 0;
        const totalConversationsWithErrors = errorCounters.get(`${_usage_counters.ONECHAT_USAGE_DOMAIN}_error_conversations_with_errors`) || 0;
        const avgErrorsPerConversation = totalConversationsWithErrors > 0 ? totalErrors / totalConversationsWithErrors : 0;
        const errorsByType = [];
        for (const [counterName, count] of Array.from(errorCounters.entries())) {
          if (count > 0) {
            if (counterName.startsWith(`${_usage_counters.ONECHAT_USAGE_DOMAIN}_error_by_type_`)) {
              const errorType = counterName.replace(`${_usage_counters.ONECHAT_USAGE_DOMAIN}_error_by_type_`, '');
              if (errorType) {
                errorsByType.push({
                  type: errorType,
                  count
                });
              }
            }
          }
        }
        errorsByType.sort((a, b) => b.count - a.count);
        const telemetry = {
          custom_tools: customTools,
          custom_agents: {
            total: customAgents
          },
          conversations,
          query_to_result_time: queryToResultTime,
          tool_calls: {
            total: totalToolCalls,
            by_source: toolCallsBySource
          },
          llm_usage: {
            by_provider: llmUsageByProvider,
            by_model: llmUsageByModel
          },
          errors: {
            total: totalErrors,
            avg_errors_per_conversation: avgErrorsPerConversation,
            total_conversations_with_errors: totalConversationsWithErrors,
            by_type: errorsByType
          }
        };
        return telemetry;
      } catch (error) {
        logger.error(`Failed to collect telemetry: ${error.message}`);
        // Return empty/default values on error
        return {
          custom_tools: {
            total: 0,
            by_type: []
          },
          custom_agents: {
            total: 0
          },
          conversations: {
            total: 0,
            total_rounds: 0,
            avg_rounds_per_conversation: 0,
            rounds_distribution: [],
            tokens_used: 0,
            average_tokens_per_conversation: 0
          },
          query_to_result_time: {
            p50: 0,
            p75: 0,
            p90: 0,
            p95: 0,
            p99: 0,
            mean: 0
          },
          tool_calls: {
            total: 0,
            by_source: {
              default_agent: 0,
              custom_agent: 0,
              mcp: 0,
              api: 0,
              a2a: 0
            }
          },
          llm_usage: {
            by_provider: [],
            by_model: []
          },
          errors: {
            total: 0,
            avg_errors_per_conversation: 0,
            total_conversations_with_errors: 0,
            by_type: []
          }
        };
      }
    }
  }));
  logger.info('Registered telemetry collector for agent_builder');
}