"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useQueryState = void 0;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _public = require("@kbn/kibana-utils-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Synchronizes state with URL query parameters.
 * Similar to `useState` but persists state in the URL for deep linking and navigation.
 *
 * @param key - The query parameter key
 * @param options.defaultValue - Default value when no state exists in URL
 * @param options.historyMode - 'replace' or 'push' for URL updates (default: 'replace')
 * @param options.parse - Function to parse the value from the URL
 * @returns [state, setState]
 */
const useQueryState = (key, {
  defaultValue,
  historyMode = 'replace',
  parse
} = {}) => {
  const history = (0, _reactRouterDom.useHistory)();
  const urlStateRef = (0, _react.useRef)((0, _public.createKbnUrlStateStorage)({
    history,
    useHash: false,
    useHashQuery: false
  }));
  const readFromUrl = (0, _react.useCallback)(() => {
    var _ref, _ref2;
    const value = urlStateRef.current.get(key);
    return (_ref = (_ref2 = parse ? parse(value) : value) !== null && _ref2 !== void 0 ? _ref2 : defaultValue) !== null && _ref !== void 0 ? _ref : null;
  }, [key, parse, defaultValue]);
  const [state, setState] = (0, _react.useState)(() => readFromUrl());
  const clearQueryParam = (0, _react.useCallback)(() => {
    const searchParams = new URLSearchParams(history.location.search);
    searchParams.delete(key);
    history.replace({
      search: searchParams.toString()
    });
  }, [history, key]);
  (0, _react.useEffect)(() => {
    if (!state) {
      clearQueryParam();
    }
  }, [state, clearQueryParam]);
  const setQueryState = (0, _react.useCallback)(async (nextValue, setOptions) => {
    setState(nextValue);
    if (nextValue) {
      var _setOptions$historyMo;
      await urlStateRef.current.set(key, nextValue, {
        replace: ((_setOptions$historyMo = setOptions === null || setOptions === void 0 ? void 0 : setOptions.historyMode) !== null && _setOptions$historyMo !== void 0 ? _setOptions$historyMo : historyMode) !== 'push'
      });
    }
  }, [historyMode, key]);
  return [state, setQueryState];
};
exports.useQueryState = useQueryState;