"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID = void 0;
exports.createRunLogRateAnalysisTool = createRunLogRateAnalysisTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _fetch_log_rate_analysis_for_alert = require("@kbn/aiops-log-rate-analysis/queries/fetch_log_rate_analysis_for_alert");
var _time = require("../../utils/time");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID = exports.OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID = 'observability.run_log_rate_analysis';
const dateRangeSchema = _zod.z.object({
  from: _zod.z.string().describe('Start of the time window expressed with Elasticsearch date math. Example: `now-15m`'),
  to: _zod.z.string().describe('End of the time window expressed with Elasticsearch date math. Example: `now`.')
});
const logRateAnalysisSchema = _zod.z.object({
  index: _zod.z.string().describe('Concrete index, data stream, or alias to analyze (for example `logs-payments.api-default`).'),
  timeFieldName: _zod.z.string().describe('Timestamp field used to build the baseline/deviation windows. Defaults to `@timestamp`.').optional(),
  baseline: dateRangeSchema.describe('Time range representing "normal" behavior that the deviation window will be compared against.'),
  deviation: dateRangeSchema.describe('Time range representing the time period with unusual behavior.'),
  searchQuery: _zod.z.record(_zod.z.any()).describe('Optional Elasticsearch query DSL filter that limits which documents are analyzed. Defaults to a match_all query.').optional()
});
function createRunLogRateAnalysisTool({
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_RUN_LOG_RATE_ANALYSIS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Identify significant changes in log rates for a given index between two time windows (baseline vs deviation) to help explain spikes or dips in log volume.`,
    schema: logRateAnalysisSchema,
    tags: ['observability', 'logs'],
    handler: async ({
      index,
      timeFieldName = '@timestamp',
      baseline,
      deviation,
      searchQuery
    }, context) => {
      try {
        const esClient = context.esClient.asCurrentUser;
        const windowParameters = {
          baselineMin: (0, _time.parseDatemath)(baseline.from),
          baselineMax: (0, _time.parseDatemath)(baseline.to, {
            roundUp: true
          }),
          deviationMin: (0, _time.parseDatemath)(deviation.from),
          deviationMax: (0, _time.parseDatemath)(deviation.to, {
            roundUp: true
          })
        };
        const response = await (0, _fetch_log_rate_analysis_for_alert.runLogRateAnalysis)({
          esClient,
          arguments: {
            index,
            windowParameters,
            timefield: timeFieldName,
            searchQuery: searchQuery !== null && searchQuery !== void 0 ? searchQuery : {
              match_all: {}
            }
          }
        });
        logger.debug(`Log rate analysis tool (index: "${index}") found ${response.significantItems.length} items of type ${response.logRateAnalysisType}.`);
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              analysisType: response.logRateAnalysisType,
              items: response.significantItems
            }
          }]
        };
      } catch (error) {
        logger.error(`Log rate analysis tool failed: ${error.message}`);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: error.message,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}