"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ElasticSearchSaver = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _langgraphCheckpoint = require("@langchain/langgraph-checkpoint");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A LangGraph checkpoint saver backed by a Elasticsearch database.
 */
class ElasticSearchSaver extends _langgraphCheckpoint.BaseCheckpointSaver {
  constructor({
    client,
    checkpointIndex,
    checkpointWritesIndex,
    refreshPolicy = 'wait_for',
    logger
  }, serde) {
    super(serde);
    (0, _defineProperty2.default)(this, "client", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "checkpointIndex", void 0);
    (0, _defineProperty2.default)(this, "checkpointWritesIndex", void 0);
    (0, _defineProperty2.default)(this, "refreshPolicy", 'wait_for');
    this.client = client;
    this.checkpointIndex = checkpointIndex !== null && checkpointIndex !== void 0 ? checkpointIndex : ElasticSearchSaver.defaultCheckpointIndex;
    this.checkpointWritesIndex = checkpointWritesIndex !== null && checkpointWritesIndex !== void 0 ? checkpointWritesIndex : ElasticSearchSaver.defaultCheckpointWritesIndex;
    this.refreshPolicy = refreshPolicy;
    this.logger = logger;
  }

  /**
   * Retrieves a checkpoint from Elasticsearch based on the
   * provided config. If the config contains a "checkpoint_id" key, the checkpoint with
   * the matching thread ID and checkpoint ID is retrieved. Otherwise, the latest checkpoint
   * for the given thread ID is retrieved.
   */
  async getTuple(config) {
    var _config$configurable, _result$hits$hits$;
    const {
      thread_id: threadId,
      checkpoint_ns: checkpointNs = '',
      checkpoint_id: checkpointId
    } = (_config$configurable = config.configurable) !== null && _config$configurable !== void 0 ? _config$configurable : {};
    const result = await this.client.search({
      index: this.checkpointIndex,
      size: 1,
      sort: [{
        checkpoint_id: {
          order: 'desc'
        }
      }],
      query: {
        bool: {
          must: [{
            term: {
              thread_id: threadId
            }
          }, {
            term: {
              checkpoint_ns: checkpointNs
            }
          }, ...(checkpointId ? [{
            term: {
              checkpoint_id: checkpointId
            }
          }] : [])]
        }
      }
    });
    if (result.hits.hits.length === 0 || ((_result$hits$hits$ = result.hits.hits[0]) === null || _result$hits$hits$ === void 0 ? void 0 : _result$hits$hits$._source) === undefined) {
      return undefined;
    }
    const doc = result.hits.hits[0]._source;
    const serializedWrites = await this.client.search({
      index: this.checkpointWritesIndex,
      sort: [{
        idx: {
          order: 'asc'
        }
      }],
      query: {
        bool: {
          must: [
          // todo(@KDKHD): fix this query to remove the duplicate should clauses. Ensure both tests and the elastic assistant work.
          {
            bool: {
              should: [{
                term: {
                  thread_id: threadId
                }
              }, {
                term: {
                  'thread_id.keyword': threadId
                }
              }],
              minimum_should_match: 1
            }
          }, {
            bool: {
              should: [{
                term: {
                  checkpoint_ns: checkpointNs
                }
              }, {
                term: {
                  'checkpoint_ns.keyword': checkpointNs
                }
              }],
              minimum_should_match: 1
            }
          }, {
            bool: {
              should: [{
                term: {
                  checkpoint_id: doc.checkpoint_id
                }
              }, {
                term: {
                  'checkpoint_id.keyword': doc.checkpoint_id
                }
              }],
              minimum_should_match: 1
            }
          }]
        }
      }
    });
    const checkpoint = await this.serde.loadsTyped(doc.type, new Uint8Array(Buffer.from(doc.checkpoint, 'base64')));
    const pendingWrites = await Promise.all(serializedWrites.hits.hits.map(async serializedWrite => {
      const source = serializedWrite._source;
      return [source.task_id, source.channel, await this.serde.loadsTyped(source.type, new Uint8Array(Buffer.from(source.value, 'base64')))];
    }));
    const configurableValues = {
      thread_id: threadId,
      checkpoint_ns: checkpointNs,
      checkpoint_id: doc.checkpoint_id
    };
    return {
      config: {
        configurable: configurableValues
      },
      checkpoint,
      pendingWrites,
      metadata: await this.serde.loadsTyped(doc.type, new Uint8Array(Buffer.from(doc.metadata, 'base64'))),
      parentConfig: doc.parent_checkpoint_id != null ? {
        configurable: {
          thread_id: threadId,
          checkpoint_ns: checkpointNs,
          checkpoint_id: doc.parent_checkpoint_id
        }
      } : undefined
    };
  }

  /**
   * Retrieve a list of checkpoint tuples from Elasticsearch based
   * on the provided config. The checkpoints are ordered by checkpoint ID
   * in descending order (newest first).
   */
  async *list(config, options) {
    var _config$configurable2, _config$configurable3, _config$configurable4;
    const {
      limit,
      before
    } = options !== null && options !== void 0 ? options : {};
    const mustClauses = [];
    if (config !== null && config !== void 0 && (_config$configurable2 = config.configurable) !== null && _config$configurable2 !== void 0 && _config$configurable2.thread_id) {
      mustClauses.push({
        term: {
          thread_id: config.configurable.thread_id
        }
      });
    }
    if ((config === null || config === void 0 ? void 0 : (_config$configurable3 = config.configurable) === null || _config$configurable3 === void 0 ? void 0 : _config$configurable3.checkpoint_ns) !== undefined && (config === null || config === void 0 ? void 0 : (_config$configurable4 = config.configurable) === null || _config$configurable4 === void 0 ? void 0 : _config$configurable4.checkpoint_ns) !== null) {
      mustClauses.push({
        term: {
          checkpoint_ns: config.configurable.checkpoint_ns
        }
      });
    }
    if (before) {
      var _before$configurable;
      mustClauses.push({
        range: {
          checkpoint_id: {
            lt: (_before$configurable = before.configurable) === null || _before$configurable === void 0 ? void 0 : _before$configurable.checkpoint_id
          }
        }
      });
    }
    const result = await this.client.search({
      index: this.checkpointIndex,
      ...(limit ? {
        size: limit
      } : {}),
      sort: [{
        checkpoint_id: {
          order: 'desc'
        }
      }, {
        '@timestamp': {
          order: 'desc'
        }
      }],
      query: {
        bool: {
          must: mustClauses
        }
      }
    });
    for await (const hit of result.hits.hits) {
      const source = hit._source;
      if (source === undefined) {
        continue;
      }
      const checkpoint = await this.serde.loadsTyped(source.type, new Uint8Array(Buffer.from(source.checkpoint, 'base64')));
      const metadata = await this.serde.loadsTyped(source.type, new Uint8Array(Buffer.from(source.metadata, 'base64')));
      yield {
        config: {
          configurable: {
            thread_id: source.thread_id,
            checkpoint_ns: source.checkpoint_ns,
            checkpoint_id: source.checkpoint_id
          }
        },
        checkpoint,
        metadata,
        parentConfig: source.parent_checkpoint_id ? {
          configurable: {
            thread_id: source.thread_id,
            checkpoint_ns: source.checkpoint_ns,
            checkpoint_id: source.parent_checkpoint_id
          }
        } : undefined
      };
    }
  }

  /**
   * Saves a checkpoint to the Elasticsearch. The checkpoint is associated
   * with the provided config and its parent config (if any).
   */
  async put(config, checkpoint, metadata) {
    var _config$configurable5, _config$configurable6, _config$configurable$, _config$configurable7, _config$configurable8;
    this.logger.debug(`Putting checkpoint ${checkpoint.id} for thread ${(_config$configurable5 = config.configurable) === null || _config$configurable5 === void 0 ? void 0 : _config$configurable5.thread_id}`);
    const threadId = (_config$configurable6 = config.configurable) === null || _config$configurable6 === void 0 ? void 0 : _config$configurable6.thread_id;
    const checkpointNs = (_config$configurable$ = (_config$configurable7 = config.configurable) === null || _config$configurable7 === void 0 ? void 0 : _config$configurable7.checkpoint_ns) !== null && _config$configurable$ !== void 0 ? _config$configurable$ : '';
    const checkpointId = checkpoint.id;
    if (threadId === undefined) {
      throw new Error(`The provided config must contain a configurable field with a "thread_id" field.`);
    }
    const [checkpointType, serializedCheckpoint] = await this.serde.dumpsTyped(checkpoint);
    const [metadataType, serializedMetadata] = await this.serde.dumpsTyped(metadata);
    if (checkpointType !== metadataType) {
      throw new Error('Mismatched checkpoint and metadata types.');
    }
    const doc = {
      '@timestamp': new Date().toISOString(),
      thread_id: threadId,
      checkpoint_ns: checkpointNs,
      checkpoint_id: checkpointId,
      parent_checkpoint_id: (_config$configurable8 = config.configurable) === null || _config$configurable8 === void 0 ? void 0 : _config$configurable8.checkpoint_id,
      type: checkpointType,
      checkpoint: Buffer.from(serializedCheckpoint).toString('base64'),
      metadata: Buffer.from(serializedMetadata).toString('base64')
    };
    const compositeId = `thread_id:${threadId}|checkpoint_ns:${checkpointNs}|checkpoint_id:${checkpointId}`;
    await this.client.update({
      index: this.checkpointIndex,
      id: compositeId,
      doc,
      doc_as_upsert: true,
      refresh: this.refreshPolicy
    });
    return {
      configurable: {
        thread_id: threadId,
        checkpoint_ns: checkpointNs,
        checkpoint_id: checkpointId
      }
    };
  }

  /**
   * Saves intermediate writes associated with a checkpoint to Elastic Search.
   */
  async putWrites(config, writes, taskId) {
    var _config$configurable9, _config$configurable10, _config$configurable11, _config$configurable12;
    this.logger.debug(`Putting writes for checkpoint ${(_config$configurable9 = config.configurable) === null || _config$configurable9 === void 0 ? void 0 : _config$configurable9.checkpoint_id}`);
    const threadId = (_config$configurable10 = config.configurable) === null || _config$configurable10 === void 0 ? void 0 : _config$configurable10.thread_id;
    const checkpointNs = (_config$configurable11 = config.configurable) === null || _config$configurable11 === void 0 ? void 0 : _config$configurable11.checkpoint_ns;
    const checkpointId = (_config$configurable12 = config.configurable) === null || _config$configurable12 === void 0 ? void 0 : _config$configurable12.checkpoint_id;
    if (threadId === undefined || checkpointNs === undefined || checkpointId === undefined) {
      throw new Error(`The provided config must contain a configurable field with "thread_id", "checkpoint_ns" and "checkpoint_id" fields.`);
    }
    const operations = (await Promise.all(writes.map(async (write, idx) => {
      const [channel, value] = write;
      const compositeId = `thread_id:${threadId}|checkpoint_ns:${checkpointNs}|checkpoint_id:${checkpointId}|task_id:${taskId}|idx:${idx}`;
      const [type, serializedValue] = await this.serde.dumpsTyped(value);
      const doc = {
        '@timestamp': new Date().toISOString(),
        thread_id: threadId,
        checkpoint_ns: checkpointNs,
        checkpoint_id: checkpointId,
        task_id: taskId,
        idx,
        channel,
        value: Buffer.from(serializedValue).toString('base64'),
        type
      };
      this.logger.debug(`Indexing write operation for checkpoint ${checkpointId}`);
      return [{
        update: {
          _index: this.checkpointWritesIndex,
          _id: compositeId
        }
      }, {
        doc,
        doc_as_upsert: true
      }];
    }))).flat();
    const result = await this.client.bulk({
      operations,
      refresh: this.refreshPolicy,
      error_trace: true
    });
    if (result.errors) {
      this.logger.error(`Failed to index writes for checkpoint ${checkpointId}`);
      throw new Error(`Failed to index writes for checkpoint ${checkpointId}`);
    }
  }

  // TODO: Implement this
  async deleteThread() {}
}
exports.ElasticSearchSaver = ElasticSearchSaver;
(0, _defineProperty2.default)(ElasticSearchSaver, "defaultCheckpointIndex", 'checkpoints');
(0, _defineProperty2.default)(ElasticSearchSaver, "defaultCheckpointWritesIndex", 'checkpoint_writes');
/**
 * When modifying the field maps, ensure you perform upgrade testing for all graphs depending on this saver.
 */
(0, _defineProperty2.default)(ElasticSearchSaver, "checkpointsFieldMap", {
  '@timestamp': {
    type: 'date',
    required: true,
    array: false
  },
  thread_id: {
    type: 'keyword',
    required: true,
    array: false
  },
  checkpoint_ns: {
    type: 'keyword',
    required: true,
    array: false
  },
  checkpoint_id: {
    type: 'keyword',
    required: false,
    array: false
  },
  parent_checkpoint_id: {
    type: 'keyword',
    required: true,
    array: false
  },
  type: {
    type: 'keyword',
    required: true,
    array: false
  },
  checkpoint: {
    type: 'binary',
    required: true,
    array: false
  },
  metadata: {
    type: 'binary',
    required: true,
    array: false
  }
});
/**
 * When modifying the field maps, ensure you perform upgrade testing for all graphs depending on this saver.
 */
(0, _defineProperty2.default)(ElasticSearchSaver, "checkpointWritesFieldMap", {
  '@timestamp': {
    type: 'date',
    required: true,
    array: false
  },
  thread_id: {
    type: 'keyword',
    required: true,
    array: false
  },
  checkpoint_ns: {
    type: 'keyword',
    required: true,
    array: false
  },
  checkpoint_id: {
    type: 'keyword',
    required: true,
    array: false
  },
  task_id: {
    type: 'keyword',
    required: true,
    array: false
  },
  idx: {
    type: 'unsigned_long',
    required: true,
    array: false
  },
  channel: {
    type: 'keyword',
    required: true,
    array: false
  },
  type: {
    type: 'keyword',
    required: true,
    array: false
  },
  value: {
    type: 'binary',
    required: true,
    array: false
  }
});