"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAgentStatusById = getAgentStatusById;
exports.getAgentStatusForAgentPolicy = getAgentStatusForAgentPolicy;
exports.getIncomingDataByAgentsId = getIncomingDataByAgentsId;
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/spaces-plugin/common");
var _constants = require("../../../common/constants");
var _services = require("../../../common/services");
var _constants2 = require("../../constants");
var _errors = require("../../errors");
var _app_context = require("../app_context");
var _helpers = require("../spaces/helpers");
var _retry = require("../epm/elasticsearch/retry");
var _agent_namespaces = require("../spaces/agent_namespaces");
var _crud = require("./crud");
var _build_status_runtime_field = require("./build_status_runtime_field");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DATA_STREAM_INDEX_PATTERN = 'logs-*-*,metrics-*-*,traces-*-*,synthetics-*-*';
const MAX_AGENT_DATA_PREVIEW_SIZE = 20;
async function getAgentStatusById(esClient, soClient, agentId) {
  return (await (0, _crud.getAgentById)(esClient, soClient, agentId)).status;
}

/**
 * getAgentStatusForAgentPolicy
 * @param esClient
 * @param soClient
 * @param agentPolicyId @deprecated use agentPolicyIds instead since the move to multi-policy
 * @param filterKuery
 * @param spaceId
 * @param agentPolicyIds
 */

async function getAgentStatusForAgentPolicy(esClient, soClient, agentPolicyId, filterKuery, spaceId, agentPolicyIds) {
  var _response, _response$aggregation, _response$aggregation2;
  const logger = _app_context.appContextService.getLogger();
  const runtimeFields = await (0, _build_status_runtime_field.buildAgentStatusRuntimeField)(soClient);
  const clauses = [];
  const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
  if (useSpaceAwareness && spaceId) {
    if (spaceId === _common.DEFAULT_SPACE_ID) {
      clauses.push((0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(_agent_namespaces.DEFAULT_NAMESPACES_FILTER)));
    } else {
      clauses.push((0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(`namespaces:"${spaceId}" or namespaces:"${_constants.ALL_SPACES_ID}"`)));
    }
  }
  if (filterKuery) {
    const kueryAsElasticsearchQuery = (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)((0, _crud.removeSOAttributes)(filterKuery)));
    clauses.push(kueryAsElasticsearchQuery);
  }
  // If agentPolicyIds is provided, we filter by those, otherwise we filter by depreciated agentPolicyId
  if (agentPolicyIds) {
    clauses.push({
      terms: {
        policy_id: agentPolicyIds
      }
    });
  } else if (agentPolicyId) {
    clauses.push({
      term: {
        policy_id: agentPolicyId
      }
    });
  }
  const query = clauses.length > 0 ? {
    bool: {
      must: clauses
    }
  } : undefined;
  const statuses = {
    online: 0,
    error: 0,
    inactive: 0,
    offline: 0,
    uninstalled: 0,
    orphaned: 0,
    updating: 0,
    unenrolled: 0,
    degraded: 0,
    enrolling: 0,
    unenrolling: 0
  };
  let response;
  try {
    response = await (0, _retry.retryTransientEsErrors)(() => esClient.search({
      index: _constants2.AGENTS_INDEX,
      size: 0,
      query,
      fields: Object.keys(runtimeFields),
      runtime_mappings: runtimeFields,
      aggregations: {
        status: {
          terms: {
            field: 'status',
            size: Object.keys(statuses).length
          }
        }
      },
      ignore_unavailable: true
    }), {
      logger
    });
  } catch (error) {
    logger.debug(`Error getting agent statuses: ${error}`);
    throw new _errors.FleetError(`Unable to retrive agent statuses for policy ${agentPolicyId} due to error: ${error.message}`);
  }
  const buckets = ((_response = response) === null || _response === void 0 ? void 0 : (_response$aggregation = _response.aggregations) === null || _response$aggregation === void 0 ? void 0 : (_response$aggregation2 = _response$aggregation.status) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets) || [];
  buckets.forEach(bucket => {
    if (statuses[bucket.key] !== undefined) {
      statuses[bucket.key] = bucket.doc_count;
    }
  });
  const {
    healthy: online,
    unhealthy: error,
    ...otherStatuses
  } = (0, _services.agentStatusesToSummary)(statuses);
  const combinedStatuses = {
    online,
    error,
    ...otherStatuses
  };
  const allStatuses = Object.values(statuses).reduce((acc, val) => acc + val, 0);
  const allActive = allStatuses - combinedStatuses.unenrolled - combinedStatuses.inactive;
  return {
    ...combinedStatuses,
    all: allStatuses,
    active: allActive,
    /* @deprecated no agents will have other status */
    other: 0,
    /* @deprecated Agent events do not exists anymore */
    events: 0,
    /* @deprecated use active instead */
    total: allActive
  };
}
async function getIncomingDataByAgentsId({
  esClient,
  agentsIds,
  dataStreamPattern = DATA_STREAM_INDEX_PATTERN,
  returnDataPreview = false
}) {
  const logger = _app_context.appContextService.getLogger();
  try {
    var _searchResult$aggrega, _searchResult$hits, _searchResult$aggrega2;
    const {
      has_all_requested: hasAllPrivileges
    } = await esClient.security.hasPrivileges({
      index: [{
        names: dataStreamPattern.split(','),
        privileges: ['read']
      }]
    });
    if (!hasAllPrivileges) {
      throw new _errors.FleetUnauthorizedError('Missing permissions to read data streams indices');
    }
    const searchResult = await (0, _retry.retryTransientEsErrors)(() => esClient.search({
      index: dataStreamPattern,
      allow_partial_search_results: true,
      _source: returnDataPreview,
      timeout: '5s',
      size: returnDataPreview ? MAX_AGENT_DATA_PREVIEW_SIZE : 0,
      query: {
        bool: {
          filter: [{
            terms: {
              'agent.id': agentsIds
            }
          }, {
            range: {
              '@timestamp': {
                gte: 'now-5m',
                lte: 'now'
              }
            }
          }]
        }
      },
      aggs: {
        agent_ids: {
          terms: {
            field: 'agent.id',
            size: agentsIds.length
          }
        }
      }
    }), {
      logger
    });
    if (!((_searchResult$aggrega = searchResult.aggregations) !== null && _searchResult$aggrega !== void 0 && _searchResult$aggrega.agent_ids)) {
      return {
        items: agentsIds.map(id => ({
          [id]: {
            data: false
          }
        })),
        dataPreview: []
      };
    }
    const dataPreview = ((_searchResult$hits = searchResult.hits) === null || _searchResult$hits === void 0 ? void 0 : _searchResult$hits.hits) || [];
    const agentIdsWithData = // @ts-expect-error aggregation type is not specified
    (_searchResult$aggrega2 = searchResult.aggregations.agent_ids.buckets.map(bucket => bucket.key)) !== null && _searchResult$aggrega2 !== void 0 ? _searchResult$aggrega2 : [];
    const items = agentsIds.map(id => agentIdsWithData.includes(id) ? {
      [id]: {
        data: true
      }
    } : {
      [id]: {
        data: false
      }
    });
    return {
      items,
      dataPreview
    };
  } catch (error) {
    logger.debug(`Error getting incoming data for agents: ${error}`);
    throw new _errors.FleetError(`Unable to retrieve incoming data for agents due to error: ${error.message}`);
  }
}