"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataOutputForAgentPolicy = getDataOutputForAgentPolicy;
exports.validateAgentPolicyOutputForIntegration = validateAgentPolicyOutputForIntegration;
exports.validateOutputForPolicy = validateOutputForPolicy;
var _output_helpers = require("../../../common/services/output_helpers");
var _constants = require("../../../common/constants");
var _services = require("../../../common/services");
var _ = require("..");
var _output = require("../output");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get the data output for a given agent policy
 * @param soClient
 * @param agentPolicy
 * @returns
 */
async function getDataOutputForAgentPolicy(soClient, agentPolicy) {
  const dataOutputId = agentPolicy.data_output_id || (await _output.outputService.getDefaultDataOutputId());
  if (!dataOutputId) {
    throw new _errors.OutputNotFoundError('No default data output found.');
  }
  return _output.outputService.get(dataOutputId);
}

/**
 * Validate outputs are valid for a policy using the current kibana licence or throw.
 * @returns
 * @param soClient
 * @param newData
 * @param existingData
 * @param allowedOutputTypeForPolicy
 */
async function validateOutputForPolicy(soClient, newData, existingData = {}, allowedOutputTypeForPolicy = Object.values(_constants.outputType)) {
  if (Object.keys(existingData).length !== 0 && newData.data_output_id === existingData.data_output_id && newData.monitoring_output_id === existingData.monitoring_output_id) {
    return;
  }
  const data = {
    ...existingData,
    ...newData
  };
  const isOutputTypeRestricted = allowedOutputTypeForPolicy.length !== Object.values(_constants.outputType).length;
  if (isOutputTypeRestricted) {
    const dataOutput = await getDataOutputForAgentPolicy(soClient, data).catch(err => {
      if (err instanceof _errors.OutputNotFoundError) {
        return;
      }
      throw err;
    });
    if (!dataOutput) {
      return;
    }
    if (!allowedOutputTypeForPolicy.includes(dataOutput.type)) {
      throw new _errors.OutputInvalidError(`Output of type "${dataOutput.type}" is not usable with policy "${data.name}".`);
    }
  }
  if (!data.data_output_id && !data.monitoring_output_id) {
    return;
  }

  // Do not validate licence output for managed and preconfigured policy
  if (data.is_managed && data.is_preconfigured) {
    return;
  }
  // Validate output when the policy has fleet server
  if ((0, _services.policyHasFleetServer)(data)) return;

  // Validate output when the policy has synthetics integration
  if ((0, _services.policyHasSyntheticsIntegration)(data)) return;
  const hasLicence = _.appContextService.getSecurityLicense().hasAtLeast(_constants.LICENCE_FOR_PER_POLICY_OUTPUT);
  if (!hasLicence) {
    throw new _errors.OutputLicenceError(`Invalid licence to set per policy output, you need ${_constants.LICENCE_FOR_PER_POLICY_OUTPUT} licence`);
  }
}
async function validateAgentPolicyOutputForIntegration(soClient, agentPolicy, packagePolicy, packageName, isNewPackagePolicy = true) {
  const allowedOutputTypeForIntegration = (0, _output_helpers.getAllowedOutputTypesForIntegration)(packageName);
  const allowedOutputTypeForPackagePolicy = (0, _output_helpers.getAllowedOutputTypesForPackagePolicy)(packagePolicy);
  const allowedOutputTypeForPolicy = allowedOutputTypeForIntegration.filter(type => allowedOutputTypeForPackagePolicy.includes(type));
  const isOutputTypeRestricted = allowedOutputTypeForPolicy.length !== Object.values(_constants.outputType).length;
  if (isOutputTypeRestricted) {
    const dataOutput = await getDataOutputForAgentPolicy(soClient, agentPolicy).catch(err => {
      if (err instanceof _errors.OutputNotFoundError) {
        return;
      }
      throw err;
    });
    if (!dataOutput) {
      return;
    }
    if (!allowedOutputTypeForPolicy.includes(dataOutput.type)) {
      if (isNewPackagePolicy) {
        throw new _errors.OutputInvalidError(`Integration "${packageName}" cannot be added to agent policy "${agentPolicy.name}" because it uses output type "${dataOutput.type}".`);
      } else {
        throw new _errors.OutputInvalidError(`Agent policy "${agentPolicy.name}" uses output type "${dataOutput.type}" which cannot be used for integration "${packageName}".`);
      }
    }
  }
}