"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.columnExists = exports.buildUserDefinedColumnsDefinitions = void 0;
exports.createInferenceEndpointToCompletionItem = createInferenceEndpointToCompletionItem;
exports.findFinalWord = void 0;
exports.findPreviousWord = findPreviousWord;
exports.getControlSuggestion = getControlSuggestion;
exports.getControlSuggestionIfSupported = getControlSuggestionIfSupported;
exports.getFieldsSuggestions = getFieldsSuggestions;
exports.getFragmentData = getFragmentData;
exports.getFunctionsSuggestions = getFunctionsSuggestions;
exports.getLastNonWhitespaceChar = getLastNonWhitespaceChar;
exports.getLiteralsSuggestions = getLiteralsSuggestions;
exports.getLookupIndexCreateSuggestion = getLookupIndexCreateSuggestion;
exports.getSafeInsertText = void 0;
exports.getValidSignaturesAndTypesToSuggestNext = getValidSignaturesAndTypesToSuggestNext;
exports.getVariablePrefix = void 0;
exports.handleFragment = handleFragment;
exports.pushItUpInTheList = pushItUpInTheList;
exports.shouldBeQuotedText = void 0;
exports.withAutoSuggest = withAutoSuggest;
exports.withinQuotes = withinQuotes;
var _esqlTypes = require("@kbn/esql-types");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _is = require("../../../ast/is");
var _types = require("../../../commands_registry/types");
var _constants = require("../../constants");
var _expressions = require("../expressions");
var _functions = require("../functions");
var _literals = require("../literals");
var _shared = require("../shared");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const shouldBeQuotedText = (text, {
  dashSupported
} = {}) => {
  return dashSupported ? /[^a-zA-Z\d_\.@-]/.test(text) : /[^a-zA-Z\d_\.@]/.test(text);
};
exports.shouldBeQuotedText = shouldBeQuotedText;
const getSafeInsertText = (text, options = {}) => {
  return shouldBeQuotedText(text, options) ? `\`${text.replace(/`/g, '``')}\`` : text;
};
exports.getSafeInsertText = getSafeInsertText;
const buildUserDefinedColumnsDefinitions = userDefinedColumns => userDefinedColumns.map(label => ({
  label,
  text: getSafeInsertText(label),
  kind: 'Variable',
  detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.variableDefinition', {
    defaultMessage: `Column specified by the user within the ES|QL query`
  }),
  sortText: 'D'
}));
exports.buildUserDefinedColumnsDefinitions = buildUserDefinedColumnsDefinitions;
function pushItUpInTheList(suggestions, shouldPromote) {
  if (!shouldPromote) {
    return suggestions;
  }
  return suggestions.map(({
    sortText,
    ...rest
  }) => ({
    ...rest,
    sortText: `1${sortText}`
  }));
}
const findFinalWord = text => {
  const words = text.split(/\s+/);
  return words[words.length - 1];
};
exports.findFinalWord = findFinalWord;
function findPreviousWord(text) {
  const words = text.split(/\s+/);
  return words[words.length - 2];
}
function withinQuotes(text) {
  const quoteCount = (text.match(/"/g) || []).length;
  return quoteCount % 2 === 1;
}

/**
 * This function handles the logic to suggest completions
 * for a given fragment of text in a generic way. A good example is
 * a field name.
 *
 * When typing a field name, there are 2 scenarios
 *
 * 1. field name is incomplete (includes the empty string)
 * KEEP /
 * KEEP fie/
 *
 * 2. field name is complete
 * KEEP field/
 *
 * This function provides a framework for detecting and handling both scenarios in a clean way.
 *
 * @param innerText - the query text before the current cursor position
 * @param isFragmentComplete — return true if the fragment is complete
 * @param getSuggestionsForIncomplete — gets suggestions for an incomplete fragment
 * @param getSuggestionsForComplete - gets suggestions for a complete fragment
 * @returns
 */
function handleFragment(innerText, isFragmentComplete, getSuggestionsForIncomplete, getSuggestionsForComplete) {
  const {
    fragment,
    rangeToReplace
  } = getFragmentData(innerText);
  if (!fragment) {
    return getSuggestionsForIncomplete('');
  } else {
    if (isFragmentComplete(fragment)) {
      return getSuggestionsForComplete(fragment, rangeToReplace);
    } else {
      return getSuggestionsForIncomplete(fragment, rangeToReplace);
    }
  }
}
function getFragmentData(innerText) {
  const fragment = findFinalWord(innerText);
  if (!fragment) {
    return {
      fragment: '',
      rangeToReplace: {
        start: 0,
        end: 0
      }
    };
  } else {
    const rangeToReplace = {
      start: innerText.length - fragment.length,
      end: innerText.length
    };
    return {
      fragment,
      rangeToReplace
    };
  }
}
async function getFieldsSuggestions(types, getFieldsByType, options = {}) {
  const {
    ignoreColumns = [],
    values = false,
    addSpaceAfterField = false,
    openSuggestions = false,
    addComma = false,
    promoteToTop = true,
    canBeMultiValue = false
  } = options;
  const variableType = (() => {
    if (canBeMultiValue) return _esqlTypes.ESQLVariableType.MULTI_VALUES;
    if (values) return _esqlTypes.ESQLVariableType.VALUES;
    return _esqlTypes.ESQLVariableType.FIELDS;
  })();
  const suggestions = await getFieldsByType(types, ignoreColumns, {
    advanceCursor: addSpaceAfterField,
    openSuggestions,
    addComma,
    variableType
  });
  return pushItUpInTheList(suggestions, promoteToTop);
}
function getFunctionsSuggestions({
  location,
  types,
  options = {},
  context,
  callbacks
}) {
  const {
    ignored = [],
    addComma = false,
    addSpaceAfterFunction = false,
    openSuggestions = false,
    constantGeneratingOnly = false
  } = options;
  const predicates = {
    location,
    returnTypes: types,
    ignored
  };
  const hasMinimumLicenseRequired = callbacks === null || callbacks === void 0 ? void 0 : callbacks.hasMinimumLicenseRequired;
  const activeProduct = context === null || context === void 0 ? void 0 : context.activeProduct;
  let filteredFunctions = (0, _functions.filterFunctionDefinitions)((0, _functions.getAllFunctions)({
    includeOperators: false
  }), predicates, hasMinimumLicenseRequired, activeProduct);

  // Filter for constant-generating functions (functions without parameters)
  if (constantGeneratingOnly) {
    const typeSet = new Set(types);
    filteredFunctions = filteredFunctions.filter(fn => fn.signatures.some(sig => sig.params.length === 0 && typeSet.has(sig.returnType)));
  }
  const textSuffix = (addComma ? ',' : '') + (addSpaceAfterFunction ? ' ' : '');
  return filteredFunctions.map(fn => {
    const suggestion = (0, _functions.getFunctionSuggestion)(fn);
    if (textSuffix) {
      suggestion.text += textSuffix;
    }
    if (openSuggestions) {
      return withAutoSuggest(suggestion);
    }
    return suggestion;
  });
}
function getLiteralsSuggestions(types, location, options = {}) {
  const {
    includeDateLiterals = true,
    includeCompatibleLiterals = true
  } = options;
  const suggestions = [];

  // Date literals gated by policy: only WHERE/EVAL/STATS_WHERE and only if types include 'date'
  if (includeDateLiterals && (location === _types.Location.WHERE || location === _types.Location.EVAL || location === _types.Location.STATS_WHERE) && types.includes('date')) {
    suggestions.push(...(0, _literals.getDateLiterals)({
      addComma: options.addComma,
      advanceCursorAndOpenSuggestions: options.advanceCursorAndOpenSuggestions
    }));
  }
  if (includeCompatibleLiterals) {
    suggestions.push(...(0, _literals.getCompatibleLiterals)(types, {
      addComma: options.addComma,
      advanceCursorAndOpenSuggestions: options.advanceCursorAndOpenSuggestions,
      supportsControls: options.supportsControls
    }, options.variables));
  }
  return suggestions;
}
function getLastNonWhitespaceChar(text) {
  return text[text.trimEnd().length - 1];
}
const columnExists = (col, context) => Boolean(context ? (0, _shared.getColumnByName)(col, context) : undefined);
exports.columnExists = columnExists;
function getControlSuggestion(type, triggerSource, variables) {
  return [{
    label: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.createControlLabel', {
      defaultMessage: 'Create control'
    }),
    text: '',
    kind: 'Issue',
    detail: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.createControlDetailLabel', {
      defaultMessage: 'Click to create'
    }),
    sortText: '1',
    command: {
      id: `esql.control.${type}.create`,
      title: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.createControlDetailLabel', {
        defaultMessage: 'Click to create'
      }),
      arguments: [{
        triggerSource
      }]
    }
  }, ...(variables !== null && variables !== void 0 && variables.length ? (0, _literals.buildConstantsDefinitions)(variables, _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.namedParamDefinition', {
    defaultMessage: 'Named parameter'
  }), '1A') : [])];
}
const getVariablePrefix = variableType => variableType === _esqlTypes.ESQLVariableType.FIELDS || variableType === _esqlTypes.ESQLVariableType.FUNCTIONS ? '??' : '?';
exports.getVariablePrefix = getVariablePrefix;
function getControlSuggestionIfSupported(supportsControls, type, triggerSource, variables, shouldBePrefixed = true) {
  var _variables$filter;
  if (!supportsControls) {
    return [];
  }
  const prefix = shouldBePrefixed ? getVariablePrefix(type) : '';
  const filteredVariables = (_variables$filter = variables === null || variables === void 0 ? void 0 : variables.filter(variable => variable.type === type)) !== null && _variables$filter !== void 0 ? _variables$filter : [];
  const controlSuggestion = getControlSuggestion(type, triggerSource, filteredVariables === null || filteredVariables === void 0 ? void 0 : filteredVariables.map(v => `${prefix}${v.key}`));
  return controlSuggestion;
}
function getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex) {
  // Filter down to signatures that match every params up to the current argIndex
  // e.g. BUCKET(longField, /) => all signatures with first param as long column type
  // or BUCKET(longField, 2, /) => all signatures with (longField, integer, ...)
  const relevantFuncSignatures = fnDefinition.signatures.filter(s => {
    var _s$params;
    return ((_s$params = s.params) === null || _s$params === void 0 ? void 0 : _s$params.length) >= argIndex && s.params.slice(0, argIndex).every(({
      type: dataType
    }, idx) => (0, _expressions.argMatchesParamType)(enrichedArgs[idx].dataType, dataType, (0, _is.isLiteral)(enrichedArgs[idx]), true));
  });
  return relevantFuncSignatures;
}

/**
 * Given a function signature, returns the compatible types to suggest for the next argument
 *
 * @param fnDefinition: the function definition
 * @param enrichedArgs: AST args with enriched esType info to match with function signatures
 * @param argIndex: the index of the argument to suggest for
 * @returns
 */
function getCompatibleParamDefs(fnDefinition, enrichedArgs, argIndex) {
  // First, narrow down to valid function signatures based on previous arguments
  const relevantFuncSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argIndex);

  // Then, get the compatible types to suggest for the next argument
  const compatibleTypesToSuggestForArg = (0, _lodash.uniqBy)(relevantFuncSignatures.map(signature => (0, _expressions.getParamAtPosition)(signature, argIndex)).filter(param => param != null), param => `${param.type}-${param.constantOnly}`);
  return compatibleTypesToSuggestForArg;
}

/**
 * Given a function signature, returns the parameter at the given position, even if it's undefined or null
 *
 * @param {params}
 * @param position
 * @returns
 */
function strictlyGetParamAtPosition({
  params
}, position) {
  return params[position] ? params[position] : null;
}
function getValidSignaturesAndTypesToSuggestNext(node, context, fnDefinition) {
  const argTypes = node.args.map(arg => (0, _expressions.getExpressionType)(arg, context === null || context === void 0 ? void 0 : context.columns));
  const enrichedArgs = node.args.map((arg, idx) => ({
    ...arg,
    dataType: argTypes[idx]
  }));

  // pick the type of the next arg
  const shouldGetNextArgument = node.text.includes(_constants.EDITOR_MARKER); // NOTE: I think this is checking if the cursor is after a comma.
  let argIndex = Math.max(node.args.length, 0);
  if (!shouldGetNextArgument && argIndex) {
    argIndex -= 1;
  }

  // For signature filtering: check ALL arguments to eliminate incompatible signatures
  // BUT only for functions with multiple signatures (overloaded functions like BUCKET)
  // For single-signature or variadic functions, use the original behavior
  const isVariadic = fnDefinition.signatures.some(sig => sig.minParams != null);
  const hasMultipleSignatures = fnDefinition.signatures.length > 1;
  const argsToCheckForFiltering = isVariadic || shouldGetNextArgument || !hasMultipleSignatures ? argIndex : enrichedArgs.length;
  const validSignatures = getValidFunctionSignaturesForPreviousArgs(fnDefinition, enrichedArgs, argsToCheckForFiltering);
  // Retrieve unique of types that are compatiable for the current arg
  const compatibleParamDefs = getCompatibleParamDefs(fnDefinition, enrichedArgs, argIndex);
  const hasMoreMandatoryArgs = !validSignatures
  // Types available to suggest next after this argument is completed
  .map(signature => strictlyGetParamAtPosition(signature, argIndex + 1))
  // when a param is null, it means param is optional
  // If there's at least one param that is optional, then
  // no need to suggest comma
  .some(p => p === null || (p === null || p === void 0 ? void 0 : p.optional) === true);
  return {
    compatibleParamDefs,
    hasMoreMandatoryArgs,
    enrichedArgs,
    argIndex,
    validSignatures
  };
}
function createInferenceEndpointToCompletionItem(inferenceEndpoint) {
  return {
    detail: _i18n.i18n.translate('kbn-esql-ast.esql.definitions.rerankInferenceIdDoc', {
      defaultMessage: 'Inference endpoint used for the completion'
    }),
    kind: 'Reference',
    label: inferenceEndpoint.inference_id,
    sortText: '1',
    text: inferenceEndpoint.inference_id
  };
}

/**
 * Given a suggestion item, decorates it with editor.action.triggerSuggest
 * that triggers the autocomplete dialog again after accepting the suggestion.
 *
 * If the suggestion item already has a custom command, it will preserve it.
 */
function withAutoSuggest(suggestionItem) {
  return {
    ...suggestionItem,
    command: suggestionItem.command ? suggestionItem.command : {
      title: 'Trigger Suggestion Dialog',
      id: 'editor.action.triggerSuggest'
    }
  };
}
function getLookupIndexCreateSuggestion(innerText, indexName) {
  const start = indexName ? innerText.lastIndexOf(indexName) : -1;
  const rangeToReplace = indexName && start !== -1 ? {
    start,
    end: start + indexName.length
  } : undefined;
  return {
    label: indexName ? _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndexWithName', {
      defaultMessage: 'Create lookup index "{indexName}"',
      values: {
        indexName
      }
    }) : _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndex', {
      defaultMessage: 'Create lookup index'
    }),
    text: indexName,
    kind: 'Issue',
    filterText: indexName,
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndexDetailLabel', {
      defaultMessage: 'Click to create'
    }),
    sortText: '0',
    command: {
      id: `esql.lookup_index.create`,
      title: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.createLookupIndexDetailLabel', {
        defaultMessage: 'Click to create'
      }),
      arguments: [{
        indexName
      }]
    },
    rangeToReplace,
    incomplete: true
  };
}