"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.esql = exports.e = void 0;
var synth = _interopRequireWildcard(require("../synth"));
var _composer_query = require("./composer_query");
var _parameter_hole = require("./parameter_hole");
var _util = require("./util");
var _is = require("../ast/is");
var _builder = require("../builder");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const esqlTag = (templateOrQueryOrParamValues, ...maybeHoles) => {
  const tagOrGeneratorWithParams = initialParamValues => (templateOrQuery, ...holes) => {
    const params = new Map(initialParamValues ? Object.entries(initialParamValues) : []);
    if (typeof templateOrQuery === 'string') {
      /**
       * Case when the query is constructed from a string, with optional
       * parameter values provided as an object.
       *
       * ```typescript
       * const query = esql('FROM index | WHERE foo > ?more', { more: 42 });
       * ```
       */
      const ast = synth.qry(templateOrQuery);
      ast.commands = (0, _util.removeNopCommands)(ast.commands);
      const moreParamValues = typeof holes[0] === 'object' && !Array.isArray(holes[0]) ? holes[0] : {};
      for (const [name, value] of Object.entries(moreParamValues)) {
        (0, _util.validateParamName)(name);
        params.set(name, value);
      }
      const query = new _composer_query.ComposerQuery(ast, params);
      return query;
    }

    /**
     * Case when the query builder is called as a tagged template:
     *
     * ```typescript
     * const query = esql `FROM index | WHERE foo > 42 | LIMIT 10`;
     * ```
     */
    const processedHoles = (0, _util.processTemplateHoles)(holes, params);
    const ast = synth.qry(templateOrQuery, ...holes);
    ast.commands = (0, _util.removeNopCommands)(ast.commands);
    const query = new _composer_query.ComposerQuery(ast, processedHoles.params);
    return query;
  };
  if (!!templateOrQueryOrParamValues && typeof templateOrQueryOrParamValues === 'object' && !Array.isArray(templateOrQueryOrParamValues)) {
    /**
     * Case when the tagged template is called with an object as the
     * first argument, which contains parameter values.
     *
     * ```typescript
     * const query = esql({ input: 42, limit: 10 }) `
     *   FROM index | WHERE foo > ?input | LIMIT ?limit`;
     * ```
     */
    return tagOrGeneratorWithParams(templateOrQueryOrParamValues);
  }

  /**
   * Case when the query composer is called as a tagged template without an
   * object as the first argument, or when called as a function with a query string.
   *
   * ```typescript
   * const query = esql `FROM index | WHERE foo > 42 | LIMIT 10`;
   * ```
   *
   * or
   *
   * ```typescript
   * const query = esql('FROM index | WHERE foo > 42 | LIMIT 10');
   * ```
   */
  return tagOrGeneratorWithParams()(templateOrQueryOrParamValues, ...maybeHoles);
};

/**
 * Creates a new {@linkcode ComposerQuery} constructor function for `FROM` or `TS`
 * command, which allows to start a new query form a list of sources and
 * optionally metadata fields.
 */
const createFromLikeStarter = cmd => (first, ...remaining) => {
  if (Array.isArray(first)) {
    return createFromSourceAndMetadataCommandStarter(cmd)(first, remaining[0]);
  } else {
    return createFromSourceCommandStarter(cmd)(first, ...remaining);
  }
};
const createFromSourceCommandStarter = cmd => (...sources) => {
  const nodes = [];
  for (const source of sources) {
    if (typeof source === 'string') {
      nodes.push(synth.src(source));
    } else if ((0, _is.isSource)(source)) {
      nodes.push(source);
    } else {
      throw new Error(`Invalid source: ${source}`);
    }
  }
  return esql`${synth.kwd(cmd)} ${nodes}`;
};
const createFromSourceAndMetadataCommandStarter = cmd => (sources, metadataFields = []) => {
  const sourceNodes = sources.map(source => {
    if (typeof source === 'string') {
      return synth.src(source);
    } else if ((0, _is.isSource)(source)) {
      return source;
    } else {
      throw new Error(`Invalid source: ${source}`);
    }
  });
  const metadataFieldsNodes = metadataFields.map(field => {
    return _builder.Builder.expression.column(field);
  });
  return metadataFieldsNodes.length ? esql`${synth.kwd(cmd)} ${sourceNodes} METADATA ${metadataFieldsNodes}` : esql`${synth.kwd(cmd)} ${sourceNodes}`;
};

/**
 * ESQL query composer tag function.
 *
 * This function allows you to create ESQL queries using template literals.
 * It supports both static and dynamic inputs, but it is designed to provide
 * safe and structured query construction with dynamic parameters.
 *
 * Basic example:
 *
 * ```typescript
 * const inputFromUser = 42;
 * const query = esql`FROM index | WHERE foo > ${inputFromUser} | LIMIT 10`;
 * ```
 *
 * @param templateOrQuery The ESQL query template or a string representing the query.
 * @param holes The dynamic values to be interpolated into the query.
 * @returns A ComposerQuery instance representing the constructed ESQL query.
 */
const esql = exports.esql = Object.assign(
/**
 * The `esql` tag function:
 *
 * ```typescript
 * const query = esql`FROM index | WHERE foo > 42 | LIMIT 10`;
 * ```
 */
esqlTag,
/**
 * Re-sharing the Synth module for convenience.
 *
 * ```typescript
 * const expression = esql.exp `abc > 42`;
 * ```
 */
synth,
/**
 * Overwriting Synth API for custom handling of ESQL parameters.
 */
{
  par: (value, name) => new _parameter_hole.ParameterHole(value, name),
  dpar: (value, name) => new _parameter_hole.DoubleParameterHole(value, name),
  from: createFromLikeStarter('FROM'),
  ts: createFromLikeStarter('TS'),
  get nop() {
    return synth.cmd`WHERE TRUE`;
  }
});

/**
 * Alias for {@link esql} for convenience.
 */
const e = exports.e = esql;